/***************************************************************************
 *   Project TupiTube Desk                                                 *
 *   Project Contact: info@tupitube.com                                    *
 *   Project Website: http://www.tupitube.com                              * 
 *                                                                         *
 *   Developers:                                                           *
 *   2025:                                                                 *
 *    Utopian Lab Development Team                                         *
 *   2010:                                                                 *
 *    Gustav Gonzalez                                                      *
 *   ---                                                                   *
 *   KTooN's versions:                                                     *
 *   2006:                                                                 *
 *    David Cuadrado                                                       *
 *    Jorge Cuadrado                                                       *
 *   2003:                                                                 *
 *    Fernado Roldan                                                       *
 *    Simena Dinas                                                         *
 *                                                                         *
 *   License:                                                              *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 *   This program is distributed in the hope that it will be useful,       *
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of        *
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the         *
 *   GNU General Public License for more details.                          *
 *                                                                         *
 *   You should have received a copy of the GNU General Public License     *
 *   along with this program.  If not, see <http://www.gnu.org/licenses/>. *
 ***************************************************************************/

#ifndef TUPPROJECT_H
#define TUPPROJECT_H

#include "tglobal.h"
#include "tupabstractserializable.h"
#include "tapplicationproperties.h"
#include "tuplibraryobject.h"

#include <QObject>
#include <QSize>

class TupScene;
class TupLayer;
class TupFrame;
class TupBackground;
class TupProjectRequest;
class TupProjectResponse;
class QGraphicsItem;
class TupLibrary;

typedef QList<TupScene *> Scenes;
typedef QList<SoundResource> Sounds;

class TUPITUBE_EXPORT TupProject : public QObject, public TupAbstractSerializable
{
    Q_OBJECT

    public:
        enum Mode 
        {
            FRAMES_MODE = 0,
            VECTOR_STATIC_BG_MODE,
            VECTOR_DYNAMIC_BG_MODE,
            RASTER_STATIC_BG_MODE,
            RASTER_DYNAMIC_BG_MODE,
            VECTOR_FG_MODE,
            NONE
        };

        TupProject(QObject *parent = nullptr);
        ~TupProject();

        void setProjectName(const QString &name);
        void setTempProjectName(const QString &tempFolder, const QString &name);
        void setAuthor(const QString &author);
        void setCurrentBgColor(const QColor &bgColor);
        void setSceneBgColor(int sceneIndex, const QColor &bgColor);
        void setDescription(const QString &desc);
        void setDimension(const QSize size);
        void setFPS(const int value, const int sceneIndex=0);
        void setVersion(const QString &value);

        QString getName() const;
        QString getAuthor() const;
        QString getDescription() const;
        QSize getDimension() const;
        QColor getCurrentBgColor() const;
        QColor getSceneBgColor(int sceneIndex) const;
        int getFPS(const int sceneIndex=0) const;
        float getVersion();

        void setDataDir(const QString &path);
        QString getDataDir() const;

        TupScene *sceneAt(int pos) const;

        int visualIndexOf(TupScene *scene) const;

        Scenes getScenes() const;
        QStringList getSceneNames();
        QList<int> getFrameLimits();

        TupScene *createScene(QString name, int pos, bool loaded = false);
        void updateScene(int pos, TupScene *scene);
        bool restoreScene(int pos);
        bool duplicateScene(int pos, const QString &sceneName);
        bool removeScene(int pos);
        bool resetScene(int pos, const QString &newName);
        QString recoverScene(int pos);
        bool moveScene(int pos, int newPos);
        TupBackground * getBackgroundFromScene(int sceneIndex);

        bool createSymbol(int type, const QString &name, const QByteArray &data, const QString &folder = QString());
        bool removeSymbol(const QString &name, TupLibraryObject::ObjectType type);
        bool addFolder(const QString &name);
        bool removeFolder(const QString &name);

        bool insertSymbolIntoFrame(TupProject::Mode spaceMode, const QString &name, int scene, int layer, int frame);
        bool removeSymbolFromFrame(const QString &name, TupLibraryObject::ObjectType type);

        bool updateSymbolId(TupLibraryObject::ObjectType type, const QString &oldId, const QString &newId);
        void reloadLibraryItem(TupLibraryObject::ObjectType type, const QString &id, TupLibraryObject *object);

        void clear();
        void loadLibrary(const QString &filename);
        void registerLibrarySounds();

        TupLibrary *getLibrary();
        void setLibrary(TupLibrary *lib);
        bool isLibraryEmpty();
        void resetSoundRecordsList();
        bool hasLibrarySounds();
        bool removeSound(const QString &name);

        void emitResponse(TupProjectResponse *response);

        virtual void fromXml(const QString &xml);
        virtual QDomElement toXml(QDomDocument &doc) const;

        void setOpen(bool open);
        bool isProjectOpen();
        int scenesCount() const;

        void updateSpaceContext(TupProject::Mode mode);
        TupProject::Mode spaceContext();

        void importLayer(int sceneIndex, const QString &layer);
        void updateRasterBackground(TupProject::Mode spaceContext, int sceneIndex, const QString &imgPath);

        void releaseLipSyncVoices(int sceneIndex, const QString &lipSyncName);

        // Sound Records API
        SoundResource getSoundResource(const QString &key);
        void addSoundResource(TupLibraryObject *object);
        bool removeSoundResource(const QString &key);
        QList<SoundResource> getSoundResourcesList() const;
        int soundsListSize();
        bool updateSoundResourcesItem(TupLibraryObject *item);
        bool updateSoundType(const QString audioId, SoundType type);
        bool updateSoundFramesToPlay(const QString audioId, int sceneIndex, QList<int> frames);
        void updateLibraryPaths(const QString &newPath);
        void updateSoundPaths(const QString &newPath);
        void swapSoundScenes(int sceneIndex, int newSceneIndex);

    signals:
        void responsed(TupProjectResponse *response);

    private:
        QString projectName;
        QString projectAuthor;
        QColor currentBgColor;
        QString projectDesc;
        QSize dimension;
        int fps;
        QString cachePath;
        float version;

        Scenes scenesList;
        Scenes undoScenes;

        int sceneCounter;
        TupLibrary *library;
        bool isOpen;
        TupProject::Mode spaceMode;

        Sounds soundRecords;
        QList<Sounds> undoSoundRecords;
};
#endif
