/***************************************************************************
 *   Project TupiTube Desk                                                 *
 *   Project Contact: info@tupitube.com                                    *
 *   Project Website: http://www.tupitube.com                              * 
 *                                                                         *
 *   Developers:                                                           *
 *   2025:                                                                 *
 *    Utopian Lab Development Team                                         *
 *   2010:                                                                 *
 *    Gustav Gonzalez                                                      *
 *   ---                                                                   *
 *   KTooN's versions:                                                     *
 *   2006:                                                                 *
 *    David Cuadrado                                                       *
 *    Jorge Cuadrado                                                       *
 *   2003:                                                                 *
 *    Fernado Roldan                                                       *
 *    Simena Dinas                                                         *
 *                                                                         *
 *   License:                                                              *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 *   This program is distributed in the hope that it will be useful,       *
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of        *
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the         *
 *   GNU General Public License for more details.                          *
 *                                                                         *
 *   You should have received a copy of the GNU General Public License     *
 *   along with this program.  If not, see <http://www.gnu.org/licenses/>. *
 ***************************************************************************/

#include "papagayoconfigurator.h"
#include "tapplicationproperties.h"
#include "tseparator.h"
#include "mouthsdialog.h"

#include <QLabel>
#include <QListWidgetItem>

PapagayoConfigurator::PapagayoConfigurator(QWidget *parent) : QFrame(parent)
{
    QBoxLayout *layout = new QBoxLayout(QBoxLayout::TopToBottom, this);
    layout->setAlignment(Qt::AlignHCenter | Qt::AlignTop);

    QLabel *toolTitle = new QLabel;
    toolTitle->setAlignment(Qt::AlignHCenter);
    QPixmap pic(THEME_DIR + "icons/papagayo.png");
    toolTitle->setPixmap(pic.scaledToWidth(20, Qt::SmoothTransformation));
    toolTitle->setToolTip(tr("Papagayo Lip-Sync Files"));
    layout->addWidget(toolTitle);
    layout->addWidget(new TSeparator(Qt::Horizontal));

    settingsLayout = new QBoxLayout(QBoxLayout::TopToBottom);
    settingsLayout->setAlignment(Qt::AlignHCenter | Qt::AlignTop);
    settingsLayout->setMargin(0);
    settingsLayout->setSpacing(0);

    setLipSyncManagerPanel();
    setPropertiesPanel();

    layout->addLayout(settingsLayout);

    mouthsWidget = new QWidget;
    QHBoxLayout *mouthsLayout = new QHBoxLayout(mouthsWidget);
    QPushButton *mouthsButton = new QPushButton(" " + tr("Mouth Samples"));
    mouthsButton->setStyleSheet("QPushButton { padding: 5px; }");
    mouthsButton->setIcon(QIcon(THEME_DIR + "icons/mouth_samples.png"));
    connect(mouthsButton, SIGNAL(clicked()), this, SLOT(openMouthsDialog()));
    mouthsLayout->addWidget(new QWidget);
    mouthsLayout->addWidget(mouthsButton);
    mouthsLayout->addWidget(new QWidget);
    layout->addWidget(mouthsWidget);

    layout->addStretch(2);
}

PapagayoConfigurator::~PapagayoConfigurator()
{
}

void PapagayoConfigurator::loadLipSyncList(QList<QString> list)
{
    manager->loadLipSyncList(list);
}

void PapagayoConfigurator::setPropertiesPanel()
{
    settingsPanel = new PapagayoSettings(this);
    connect(settingsPanel, SIGNAL(selectMouth(const QString&,int)), this, SIGNAL(selectMouth(const QString&,int)));
    connect(settingsPanel, SIGNAL(closeLipSyncProperties()), this, SLOT(closeSettingsPanel()));
    connect(settingsPanel, SIGNAL(initFrameHasChanged(int)), this, SIGNAL(initFrameHasChanged(int)));
    connect(settingsPanel, SIGNAL(xPosChanged(int)), this, SIGNAL(xPosChanged(int)));
    connect(settingsPanel, SIGNAL(yPosChanged(int)), this, SIGNAL(yPosChanged(int)));
    connect(settingsPanel, SIGNAL(rotationChanged(int)), this, SIGNAL(rotationChanged(int)));
    connect(settingsPanel, SIGNAL(scaleChanged(double,double)), this, SIGNAL(scaleChanged(double,double)));

    connect(settingsPanel, SIGNAL(objectHasBeenReset()), this, SIGNAL(objectHasBeenReset()));
    connect(settingsPanel, SIGNAL(proportionActivated(bool)), this, SIGNAL(proportionActivated(bool)));
    connect(settingsPanel, SIGNAL(notifyForwardFlagUpdated(int)), this, SIGNAL(notifyForwardFlagUpdated(int)));

    /* SQA: These connections don't work on Windows
    connect(settingsPanel, &PapagayoSettings::selectMouth, this, &PapagayoConfigurator::selectMouth);
    connect(settingsPanel, &PapagayoSettings::closeLipSyncProperties, this, &PapagayoConfigurator::closeSettingsPanel);
    connect(settingsPanel, &PapagayoSettings::initFrameHasChanged, this, &PapagayoConfigurator::initFrameHasChanged);
    connect(settingsPanel, &PapagayoSettings::xPosChanged, this, &PapagayoConfigurator::xPosChanged);
    connect(settingsPanel, &PapagayoSettings::yPosChanged, this, &PapagayoConfigurator::yPosChanged);
    connect(settingsPanel, &PapagayoSettings::rotationChanged, this, &PapagayoConfigurator::rotationChanged);
    connect(settingsPanel, &PapagayoSettings::scaleChanged, this, &PapagayoConfigurator::scaleChanged);

    connect(settingsPanel, &PapagayoSettings::objectHasBeenReset, this, &PapagayoConfigurator::objectHasBeenReset);
    connect(settingsPanel, &PapagayoSettings::proportionActivated, this, &PapagayoConfigurator::proportionActivated);
    */

    settingsLayout->addWidget(settingsPanel);

    activePropertiesPanel(false);
}

void PapagayoConfigurator::activePropertiesPanel(bool enable)
{
    if (enable)
        settingsPanel->show();
    else
        settingsPanel->hide();
}

void PapagayoConfigurator::setLipSyncManagerPanel()
{
    manager = new LipSyncManager(this);
    connect(manager, SIGNAL(lipsyncCreatorRequested()), this, SIGNAL(lipsyncCreatorRequested()));
    connect(manager, SIGNAL(lipsyncEditionRequested(const QString&)), this, SIGNAL(lipsyncEditionRequested(const QString&)));
    connect(manager, SIGNAL(mouthEditionRequested(const QString&)), this, SLOT(editCurrentLipSync(const QString&)));
    connect(manager, SIGNAL(currentLipSyncRemoved(const QString&)), this, SIGNAL(currentLipsyncRemoved(const QString&)));

    /* SQA: These connections don't work on Windows
    connect(manager, &LipSyncManager::lipsyncCreatorRequested, this, &PapagayoConfigurator::lipsyncCreatorRequested);
    connect(manager, &LipSyncManager::lipsyncEditionRequested, this, &PapagayoConfigurator::lipsyncEditionRequested);
    connect(manager, &LipSyncManager::mouthEditionRequested, this, &PapagayoConfigurator::editCurrentLipSync);
    connect(manager, &LipSyncManager::currentLipSyncRemoved, this, &PapagayoConfigurator::currentLipsyncRemoved);
    */

    settingsLayout->addWidget(manager);
}

void PapagayoConfigurator::activeLipSyncManagerPanel(bool enable)
{
    if (enable)
        manager->show();
    else
        manager->hide();
}

void PapagayoConfigurator::addLipSyncRecord(const QString &name)
{
    manager->addNewRecord(name);
}

void PapagayoConfigurator::removeLipSyncRecord(const QString &name)
{
    manager->removeRecordFromList(name);
}

void PapagayoConfigurator::editCurrentLipSync(const QString &name)
{
    emit mouthEditionRequested(name);

    activeLipSyncManagerPanel(false);
    mouthsWidget->setVisible(false);
    activePropertiesPanel(true);
}

void PapagayoConfigurator::openLipSyncProperties(TupLipSync *lipsync)
{
    settingsPanel->openLipSyncProperties(lipsync);
}

void PapagayoConfigurator::resetUI()
{
    manager->resetUI();
    closeSettingsPanel();
}

void PapagayoConfigurator::closeSettingsPanel()
{
    emit closeLipSyncProperties();
    closePanels();
}

void PapagayoConfigurator::closePanels()
{
    activeLipSyncManagerPanel(true);
    mouthsWidget->setVisible(true);
    activePropertiesPanel(false);
}

void PapagayoConfigurator::updateInterfaceRecords()
{
    settingsPanel->updateInterfaceRecords();
}

void PapagayoConfigurator::setTransformations(const QDomElement &dom)
{
    settingsPanel->setTransformations(dom);
}

void PapagayoConfigurator::setTransformations(const TupTransformation::Parameters parameters)
{
    settingsPanel->setTransformations(parameters);
}

void PapagayoConfigurator::updatePositionCoords(int x, int y)
{
    settingsPanel->updatePositionCoords(x, y);
}

void PapagayoConfigurator::updateRotationAngle(int angle)
{
    settingsPanel->updateRotationAngle(angle);
}

void PapagayoConfigurator::updateScaleFactor(double x, double y)
{
    settingsPanel->updateScaleFactor(x, y);
}

void PapagayoConfigurator::setPhoneme(const TupPhoneme *phoneme)
{
    settingsPanel->setPhoneme(phoneme);
}

void PapagayoConfigurator::openMouthsDialog()
{
    MouthsDialog *dialog = new MouthsDialog();
    dialog->show();
}

void PapagayoConfigurator::setProportionState(bool flag)
{
    settingsPanel->setProportionState(flag);
}
