/*
 * Copyright LWJGL. All rights reserved.
 * License terms: https://www.lwjgl.org/license
 * MACHINE GENERATED FILE, DO NOT EDIT
 */
package org.lwjgl.vulkan;

import org.jspecify.annotations.*;

import java.nio.*;

import org.lwjgl.system.*;

import static org.lwjgl.system.Checks.*;
import static org.lwjgl.system.JNI.*;
import static org.lwjgl.system.MemoryUtil.*;

/**
 * This extension adds transform feedback to the Vulkan API by exposing the SPIR-V {@code TransformFeedback} and {@code GeometryStreams} capabilities to capture vertex, tessellation or geometry shader outputs to one or more buffers. It adds API functionality to bind transform feedback buffers to capture the primitives emitted by the graphics pipeline from SPIR-V outputs decorated for transform feedback. The transform feedback capture can be paused and resumed by way of storing and retrieving a byte counter. The captured data can be drawn again where the vertex count is derived from the byte counter without CPU intervention. If the implementation is capable, a vertex stream other than zero can be rasterized.
 * 
 * <p>All these features are designed to match the full capabilities of OpenGL core transform feedback functionality and beyond. Many of the features are optional to allow base OpenGL ES GPUs to also implement this extension.</p>
 * 
 * <p>The primary purpose of the functionality exposed by this extension is to support translation layers from other 3D APIs. This functionality is not considered forward looking, and is not expected to be promoted to a KHR extension or to core Vulkan. Unless this is needed for translation, it is recommended that developers use alternative techniques of using the GPU to process and capture vertex data.</p>
 * 
 * <dl>
 * <dt><b>Name String</b></dt>
 * <dd>{@code VK_EXT_transform_feedback}</dd>
 * <dt><b>Extension Type</b></dt>
 * <dd>Device extension</dd>
 * <dt><b>Registered Extension Number</b></dt>
 * <dd>29</dd>
 * <dt><b>Revision</b></dt>
 * <dd>1</dd>
 * <dt><b>Extension and Version Dependencies</b></dt>
 * <dd>{@link KHRGetPhysicalDeviceProperties2 VK_KHR_get_physical_device_properties2} or <a href="https://registry.khronos.org/vulkan/specs/latest/html/vkspec.html#versions-1.1">Version 1.1</a></dd>
 * <dt><b>Special Uses</b></dt>
 * <dd><ul>
 * <li><a href="https://registry.khronos.org/vulkan/specs/latest/html/vkspec.html#extendingvulkan-compatibility-specialuse">OpenGL / ES support</a></li>
 * <li><a href="https://registry.khronos.org/vulkan/specs/latest/html/vkspec.html#extendingvulkan-compatibility-specialuse">D3D support</a></li>
 * <li><a href="https://registry.khronos.org/vulkan/specs/latest/html/vkspec.html#extendingvulkan-compatibility-specialuse">Developer tools</a></li>
 * </ul></dd>
 * <dt><b>Contact</b></dt>
 * <dd><ul>
 * <li>Piers Daniell <a href="https://github.com/KhronosGroup/Vulkan-Docs/issues/new?body=[VK_EXT_transform_feedback]%20@pdaniell-nv%250A*Here%20describe%20the%20issue%20or%20question%20you%20have%20about%20the%20VK_EXT_transform_feedback%20extension*">pdaniell-nv</a></li>
 * </ul></dd>
 * </dl>
 * 
 * <h5>Other Extension Metadata</h5>
 * 
 * <dl>
 * <dt><b>Last Modified Date</b></dt>
 * <dd>2018-10-09</dd>
 * <dt><b>Contributors</b></dt>
 * <dd><ul>
 * <li>Baldur Karlsson, Valve</li>
 * <li>Boris Zanin, Mobica</li>
 * <li>Daniel Rakos, AMD</li>
 * <li>Donald Scorgie, Imagination</li>
 * <li>Henri Verbeet, CodeWeavers</li>
 * <li>Jan-Harald Fredriksen, Arm</li>
 * <li>Faith Ekstrand, Intel</li>
 * <li>Jeff Bolz, NVIDIA</li>
 * <li>Jesse Barker, Unity</li>
 * <li>Jesse Hall, Google</li>
 * <li>Pierre-Loup Griffais, Valve</li>
 * <li>Philip Rebohle, DXVK</li>
 * <li>Ruihao Zhang, Qualcomm</li>
 * <li>Samuel Pitoiset, Valve</li>
 * <li>Slawomir Grajewski, Intel</li>
 * <li>Stu Smith, Imagination Technologies</li>
 * </ul></dd>
 * </dl>
 */
public class EXTTransformFeedback {

    /** The extension specification version. */
    public static final int VK_EXT_TRANSFORM_FEEDBACK_SPEC_VERSION = 1;

    /** The extension name. */
    public static final String VK_EXT_TRANSFORM_FEEDBACK_EXTENSION_NAME = "VK_EXT_transform_feedback";

    /**
     * Extends {@code VkStructureType}.
     * 
     * <h5>Enum values:</h5>
     * 
     * <ul>
     * <li>{@link #VK_STRUCTURE_TYPE_PHYSICAL_DEVICE_TRANSFORM_FEEDBACK_FEATURES_EXT STRUCTURE_TYPE_PHYSICAL_DEVICE_TRANSFORM_FEEDBACK_FEATURES_EXT}</li>
     * <li>{@link #VK_STRUCTURE_TYPE_PHYSICAL_DEVICE_TRANSFORM_FEEDBACK_PROPERTIES_EXT STRUCTURE_TYPE_PHYSICAL_DEVICE_TRANSFORM_FEEDBACK_PROPERTIES_EXT}</li>
     * <li>{@link #VK_STRUCTURE_TYPE_PIPELINE_RASTERIZATION_STATE_STREAM_CREATE_INFO_EXT STRUCTURE_TYPE_PIPELINE_RASTERIZATION_STATE_STREAM_CREATE_INFO_EXT}</li>
     * </ul>
     */
    public static final int
        VK_STRUCTURE_TYPE_PHYSICAL_DEVICE_TRANSFORM_FEEDBACK_FEATURES_EXT     = 1000028000,
        VK_STRUCTURE_TYPE_PHYSICAL_DEVICE_TRANSFORM_FEEDBACK_PROPERTIES_EXT   = 1000028001,
        VK_STRUCTURE_TYPE_PIPELINE_RASTERIZATION_STATE_STREAM_CREATE_INFO_EXT = 1000028002;

    /** Extends {@code VkQueryType}. */
    public static final int VK_QUERY_TYPE_TRANSFORM_FEEDBACK_STREAM_EXT = 1000028004;

    /**
     * Extends {@code VkBufferUsageFlagBits}.
     * 
     * <h5>Enum values:</h5>
     * 
     * <ul>
     * <li>{@link #VK_BUFFER_USAGE_TRANSFORM_FEEDBACK_BUFFER_BIT_EXT BUFFER_USAGE_TRANSFORM_FEEDBACK_BUFFER_BIT_EXT}</li>
     * <li>{@link #VK_BUFFER_USAGE_TRANSFORM_FEEDBACK_COUNTER_BUFFER_BIT_EXT BUFFER_USAGE_TRANSFORM_FEEDBACK_COUNTER_BUFFER_BIT_EXT}</li>
     * </ul>
     */
    public static final int
        VK_BUFFER_USAGE_TRANSFORM_FEEDBACK_BUFFER_BIT_EXT         = 0x800,
        VK_BUFFER_USAGE_TRANSFORM_FEEDBACK_COUNTER_BUFFER_BIT_EXT = 0x1000;

    /**
     * Extends {@code VkAccessFlagBits}.
     * 
     * <h5>Enum values:</h5>
     * 
     * <ul>
     * <li>{@link #VK_ACCESS_TRANSFORM_FEEDBACK_WRITE_BIT_EXT ACCESS_TRANSFORM_FEEDBACK_WRITE_BIT_EXT}</li>
     * <li>{@link #VK_ACCESS_TRANSFORM_FEEDBACK_COUNTER_READ_BIT_EXT ACCESS_TRANSFORM_FEEDBACK_COUNTER_READ_BIT_EXT}</li>
     * <li>{@link #VK_ACCESS_TRANSFORM_FEEDBACK_COUNTER_WRITE_BIT_EXT ACCESS_TRANSFORM_FEEDBACK_COUNTER_WRITE_BIT_EXT}</li>
     * </ul>
     */
    public static final int
        VK_ACCESS_TRANSFORM_FEEDBACK_WRITE_BIT_EXT         = 0x2000000,
        VK_ACCESS_TRANSFORM_FEEDBACK_COUNTER_READ_BIT_EXT  = 0x4000000,
        VK_ACCESS_TRANSFORM_FEEDBACK_COUNTER_WRITE_BIT_EXT = 0x8000000;

    /** Extends {@code VkPipelineStageFlagBits}. */
    public static final int VK_PIPELINE_STAGE_TRANSFORM_FEEDBACK_BIT_EXT = 0x1000000;

    protected EXTTransformFeedback() {
        throw new UnsupportedOperationException();
    }

    // --- [ vkCmdBindTransformFeedbackBuffersEXT ] ---

    /**
     * Unsafe version of: {@link #vkCmdBindTransformFeedbackBuffersEXT CmdBindTransformFeedbackBuffersEXT}
     *
     * @param bindingCount the number of transform feedback bindings whose state is updated by the command.
     */
    public static void nvkCmdBindTransformFeedbackBuffersEXT(VkCommandBuffer commandBuffer, int firstBinding, int bindingCount, long pBuffers, long pOffsets, long pSizes) {
        long __functionAddress = commandBuffer.getCapabilities().vkCmdBindTransformFeedbackBuffersEXT;
        if (CHECKS) {
            check(__functionAddress);
        }
        callPPPPV(commandBuffer.address(), firstBinding, bindingCount, pBuffers, pOffsets, pSizes, __functionAddress);
    }

    /**
     * Bind transform feedback buffers to a command buffer.
     * 
     * <h5>C Specification</h5>
     * 
     * <p>To bind transform feedback buffers to a command buffer for use in subsequent drawing commands, call:</p>
     * 
     * <pre><code>
     * void vkCmdBindTransformFeedbackBuffersEXT(
     *     VkCommandBuffer                             commandBuffer,
     *     uint32_t                                    firstBinding,
     *     uint32_t                                    bindingCount,
     *     const VkBuffer*                             pBuffers,
     *     const VkDeviceSize*                         pOffsets,
     *     const VkDeviceSize*                         pSizes);</code></pre>
     * 
     * <h5>Description</h5>
     * 
     * <p>The values taken from elements <code>i</code> of {@code pBuffers}, {@code pOffsets} and {@code pSizes} replace the current state for the transform feedback binding <code>firstBinding + i</code>, for <code>i</code> in <code>[0, bindingCount)</code>. The transform feedback binding is updated to start at the offset indicated by {@code pOffsets}[i] from the start of the buffer {@code pBuffers}[i].</p>
     * 
     * <h5>Valid Usage</h5>
     * 
     * <ul>
     * <li>{@link VkPhysicalDeviceTransformFeedbackFeaturesEXT}{@code ::transformFeedback} <b>must</b> be enabled</li>
     * <li>{@code firstBinding} <b>must</b> be less than {@link VkPhysicalDeviceTransformFeedbackPropertiesEXT}{@code ::maxTransformFeedbackBuffers}</li>
     * <li>The sum of {@code firstBinding} and {@code bindingCount} <b>must</b> be less than or equal to {@link VkPhysicalDeviceTransformFeedbackPropertiesEXT}{@code ::maxTransformFeedbackBuffers}</li>
     * <li>All elements of {@code pOffsets} <b>must</b> be less than the size of the corresponding element in {@code pBuffers}</li>
     * <li>All elements of {@code pOffsets} <b>must</b> be a multiple of 4</li>
     * <li>All elements of {@code pBuffers} <b>must</b> have been created with the {@link #VK_BUFFER_USAGE_TRANSFORM_FEEDBACK_BUFFER_BIT_EXT BUFFER_USAGE_TRANSFORM_FEEDBACK_BUFFER_BIT_EXT} flag</li>
     * <li>If the optional {@code pSize} array is specified, each element of {@code pSizes} <b>must</b> either be {@link VK10#VK_WHOLE_SIZE WHOLE_SIZE}, or be less than or equal to {@link VkPhysicalDeviceTransformFeedbackPropertiesEXT}{@code ::maxTransformFeedbackBufferSize}</li>
     * <li>All elements of {@code pSizes} <b>must</b> be either {@link VK10#VK_WHOLE_SIZE WHOLE_SIZE}, or less than or equal to the size of the corresponding buffer in {@code pBuffers}</li>
     * <li>All elements of {@code pOffsets} plus {@code pSizes}, where the {@code pSizes}, element is not {@link VK10#VK_WHOLE_SIZE WHOLE_SIZE}, <b>must</b> be less than or equal to the size of the corresponding buffer in {@code pBuffers}</li>
     * <li>Each element of {@code pBuffers} that is non-sparse <b>must</b> be bound completely and contiguously to a single {@code VkDeviceMemory} object</li>
     * <li>Transform feedback <b>must</b> not be active when the {@code vkCmdBindTransformFeedbackBuffersEXT} command is recorded</li>
     * </ul>
     * 
     * <h5>Valid Usage (Implicit)</h5>
     * 
     * <ul>
     * <li>{@code commandBuffer} <b>must</b> be a valid {@code VkCommandBuffer} handle</li>
     * <li>{@code pBuffers} <b>must</b> be a valid pointer to an array of {@code bindingCount} valid {@code VkBuffer} handles</li>
     * <li>{@code pOffsets} <b>must</b> be a valid pointer to an array of {@code bindingCount} {@code VkDeviceSize} values</li>
     * <li>{@code commandBuffer} <b>must</b> be in the <a href="https://registry.khronos.org/vulkan/specs/latest/html/vkspec.html#commandbuffers-lifecycle">recording state</a></li>
     * <li>The {@code VkCommandPool} that {@code commandBuffer} was allocated from <b>must</b> support graphics operations</li>
     * <li>This command <b>must</b> only be called outside of a video coding scope</li>
     * <li>{@code bindingCount} <b>must</b> be greater than 0</li>
     * <li>Both of {@code commandBuffer}, and the elements of {@code pBuffers} <b>must</b> have been created, allocated, or retrieved from the same {@code VkDevice}</li>
     * </ul>
     * 
     * <h5>Host Synchronization</h5>
     * 
     * <ul>
     * <li>Host access to {@code commandBuffer} <b>must</b> be externally synchronized</li>
     * <li>Host access to the {@code VkCommandPool} that {@code commandBuffer} was allocated from <b>must</b> be externally synchronized</li>
     * </ul>
     * 
     * <h5>Command Properties</h5>
     * 
     * <table class="lwjgl">
     * <thead><tr><th><a href="https://registry.khronos.org/vulkan/specs/latest/html/vkspec.html#VkCommandBufferLevel">Command Buffer Levels</a></th><th><a href="https://registry.khronos.org/vulkan/specs/latest/html/vkspec.html#vkCmdBeginRenderPass">Render Pass Scope</a></th><th><a href="https://registry.khronos.org/vulkan/specs/latest/html/vkspec.html#vkCmdBeginVideoCodingKHR">Video Coding Scope</a></th><th><a href="https://registry.khronos.org/vulkan/specs/latest/html/vkspec.html#VkQueueFlagBits">Supported Queue Types</a></th><th><a href="https://registry.khronos.org/vulkan/specs/latest/html/vkspec.html#fundamentals-queueoperation-command-types">Command Type</a></th></tr></thead>
     * <tbody><tr><td>Primary Secondary</td><td>Both</td><td>Outside</td><td>Graphics</td><td>State</td></tr></tbody>
     * </table>
     *
     * @param commandBuffer the command buffer into which the command is recorded.
     * @param firstBinding  the index of the first transform feedback binding whose state is updated by the command.
     * @param pBuffers      a pointer to an array of buffer handles.
     * @param pOffsets      a pointer to an array of buffer offsets.
     * @param pSizes        {@code NULL} or a pointer to an array of {@code VkDeviceSize} buffer sizes, specifying the maximum number of bytes to capture to the corresponding transform feedback buffer. If {@code pSizes} is {@code NULL}, or the value of the {@code pSizes} array element is {@link VK10#VK_WHOLE_SIZE WHOLE_SIZE}, then the maximum number of bytes captured will be the size of the corresponding buffer minus the buffer offset.
     */
    public static void vkCmdBindTransformFeedbackBuffersEXT(VkCommandBuffer commandBuffer, @NativeType("uint32_t") int firstBinding, @NativeType("VkBuffer const *") LongBuffer pBuffers, @NativeType("VkDeviceSize const *") LongBuffer pOffsets, @NativeType("VkDeviceSize const *") @Nullable LongBuffer pSizes) {
        if (CHECKS) {
            check(pOffsets, pBuffers.remaining());
            checkSafe(pSizes, pBuffers.remaining());
        }
        nvkCmdBindTransformFeedbackBuffersEXT(commandBuffer, firstBinding, pBuffers.remaining(), memAddress(pBuffers), memAddress(pOffsets), memAddressSafe(pSizes));
    }

    // --- [ vkCmdBeginTransformFeedbackEXT ] ---

    /**
     * Unsafe version of: {@link #vkCmdBeginTransformFeedbackEXT CmdBeginTransformFeedbackEXT}
     *
     * @param counterBufferCount the size of the {@code pCounterBuffers} and {@code pCounterBufferOffsets} arrays.
     */
    public static void nvkCmdBeginTransformFeedbackEXT(VkCommandBuffer commandBuffer, int firstCounterBuffer, int counterBufferCount, long pCounterBuffers, long pCounterBufferOffsets) {
        long __functionAddress = commandBuffer.getCapabilities().vkCmdBeginTransformFeedbackEXT;
        if (CHECKS) {
            check(__functionAddress);
        }
        callPPPV(commandBuffer.address(), firstCounterBuffer, counterBufferCount, pCounterBuffers, pCounterBufferOffsets, __functionAddress);
    }

    /**
     * Make transform feedback active in the command buffer.
     * 
     * <h5>C Specification</h5>
     * 
     * <p>Transform feedback for specific transform feedback buffers is made active by calling:</p>
     * 
     * <pre><code>
     * void vkCmdBeginTransformFeedbackEXT(
     *     VkCommandBuffer                             commandBuffer,
     *     uint32_t                                    firstCounterBuffer,
     *     uint32_t                                    counterBufferCount,
     *     const VkBuffer*                             pCounterBuffers,
     *     const VkDeviceSize*                         pCounterBufferOffsets);</code></pre>
     * 
     * <h5>Description</h5>
     * 
     * <p>The active transform feedback buffers will capture primitives emitted from the corresponding {@code XfbBuffer} in the bound graphics pipeline. Any {@code XfbBuffer} emitted that does not output to an active transform feedback buffer will not be captured.</p>
     * 
     * <h5>Valid Usage</h5>
     * 
     * <ul>
     * <li>{@link VkPhysicalDeviceTransformFeedbackFeaturesEXT}{@code ::transformFeedback} <b>must</b> be enabled</li>
     * <li>Transform feedback <b>must</b> not be active</li>
     * <li>{@code firstCounterBuffer} <b>must</b> be less than {@link VkPhysicalDeviceTransformFeedbackPropertiesEXT}{@code ::maxTransformFeedbackBuffers}</li>
     * <li>The sum of {@code firstCounterBuffer} and {@code counterBufferCount} <b>must</b> be less than or equal to {@link VkPhysicalDeviceTransformFeedbackPropertiesEXT}{@code ::maxTransformFeedbackBuffers}</li>
     * <li>If {@code counterBufferCount} is not 0, and {@code pCounterBuffers} is not {@code NULL}, {@code pCounterBuffers} <b>must</b> be a valid pointer to an array of {@code counterBufferCount} {@code VkBuffer} handles that are either valid or {@link VK10#VK_NULL_HANDLE NULL_HANDLE}</li>
     * <li>For each buffer handle in the array, if it is not {@link VK10#VK_NULL_HANDLE NULL_HANDLE} it <b>must</b> reference a buffer large enough to hold 4 bytes at the corresponding offset from the {@code pCounterBufferOffsets} array</li>
     * <li>If {@code pCounterBuffer} is {@code NULL}, then {@code pCounterBufferOffsets} <b>must</b> also be {@code NULL}</li>
     * <li>For each buffer handle in the {@code pCounterBuffers} array that is not {@link VK10#VK_NULL_HANDLE NULL_HANDLE} it <b>must</b> have been created with a {@code usage} value containing {@link #VK_BUFFER_USAGE_TRANSFORM_FEEDBACK_COUNTER_BUFFER_BIT_EXT BUFFER_USAGE_TRANSFORM_FEEDBACK_COUNTER_BUFFER_BIT_EXT}</li>
     * <li>The sum of {@code firstCounterBuffer} and {@code counterBufferCount} <b>must</b> be less than or equal to the number of transform feedback buffers bound by {@link #vkCmdBindTransformFeedbackBuffersEXT CmdBindTransformFeedbackBuffersEXT}</li>
     * <li>If the <a href="https://registry.khronos.org/vulkan/specs/latest/html/vkspec.html#features-shaderObject">{@code shaderObject}</a> feature is not enabled, a valid graphics pipeline <b>must</b> be bound to {@link VK10#VK_PIPELINE_BIND_POINT_GRAPHICS PIPELINE_BIND_POINT_GRAPHICS}</li>
     * <li>The last <a href="https://registry.khronos.org/vulkan/specs/latest/html/vkspec.html#pipelines-graphics-subsets-pre-rasterization">pre-rasterization shader stage</a> of the bound graphics pipeline <b>must</b> have been declared with the {@code Xfb} execution mode</li>
     * <li>Transform feedback <b>must</b> not be made active in a render pass instance with multiview enabled</li>
     * </ul>
     * 
     * <h5>Valid Usage (Implicit)</h5>
     * 
     * <ul>
     * <li>{@code commandBuffer} <b>must</b> be a valid {@code VkCommandBuffer} handle</li>
     * <li>If {@code counterBufferCount} is not 0, and {@code pCounterBufferOffsets} is not {@code NULL}, {@code pCounterBufferOffsets} <b>must</b> be a valid pointer to an array of {@code counterBufferCount} {@code VkDeviceSize} values</li>
     * <li>{@code commandBuffer} <b>must</b> be in the <a href="https://registry.khronos.org/vulkan/specs/latest/html/vkspec.html#commandbuffers-lifecycle">recording state</a></li>
     * <li>The {@code VkCommandPool} that {@code commandBuffer} was allocated from <b>must</b> support graphics operations</li>
     * <li>This command <b>must</b> only be called inside of a render pass instance</li>
     * <li>This command <b>must</b> only be called outside of a video coding scope</li>
     * <li>Both of {@code commandBuffer}, and the elements of {@code pCounterBuffers} that are valid handles of non-ignored parameters <b>must</b> have been created, allocated, or retrieved from the same {@code VkDevice}</li>
     * </ul>
     * 
     * <h5>Host Synchronization</h5>
     * 
     * <ul>
     * <li>Host access to {@code commandBuffer} <b>must</b> be externally synchronized</li>
     * <li>Host access to the {@code VkCommandPool} that {@code commandBuffer} was allocated from <b>must</b> be externally synchronized</li>
     * </ul>
     * 
     * <h5>Command Properties</h5>
     * 
     * <table class="lwjgl">
     * <thead><tr><th><a href="https://registry.khronos.org/vulkan/specs/latest/html/vkspec.html#VkCommandBufferLevel">Command Buffer Levels</a></th><th><a href="https://registry.khronos.org/vulkan/specs/latest/html/vkspec.html#vkCmdBeginRenderPass">Render Pass Scope</a></th><th><a href="https://registry.khronos.org/vulkan/specs/latest/html/vkspec.html#vkCmdBeginVideoCodingKHR">Video Coding Scope</a></th><th><a href="https://registry.khronos.org/vulkan/specs/latest/html/vkspec.html#VkQueueFlagBits">Supported Queue Types</a></th><th><a href="https://registry.khronos.org/vulkan/specs/latest/html/vkspec.html#fundamentals-queueoperation-command-types">Command Type</a></th></tr></thead>
     * <tbody><tr><td>Primary Secondary</td><td>Inside</td><td>Outside</td><td>Graphics</td><td>State</td></tr></tbody>
     * </table>
     *
     * @param commandBuffer         the command buffer into which the command is recorded.
     * @param firstCounterBuffer    the index of the first transform feedback buffer corresponding to {@code pCounterBuffers}[0] and {@code pCounterBufferOffsets}[0].
     * @param pCounterBuffers       {@code NULL} or a pointer to an array of {@code VkBuffer} handles to counter buffers. Each buffer contains a 4 byte integer value representing the byte offset from the start of the corresponding transform feedback buffer from where to start capturing vertex data. If the byte offset stored to the counter buffer location was done using {@link #vkCmdEndTransformFeedbackEXT CmdEndTransformFeedbackEXT} it can be used to resume transform feedback from the previous location. If {@code pCounterBuffers} is {@code NULL}, then transform feedback will start capturing vertex data to byte offset zero in all bound transform feedback buffers. For each element of {@code pCounterBuffers} that is {@link VK10#VK_NULL_HANDLE NULL_HANDLE}, transform feedback will start capturing vertex data to byte zero in the corresponding bound transform feedback buffer.
     * @param pCounterBufferOffsets {@code NULL} or a pointer to an array of {@code VkDeviceSize} values specifying offsets within each of the {@code pCounterBuffers} where the counter values were previously written. The location in each counter buffer at these offsets <b>must</b> be large enough to contain 4 bytes of data. This data is the number of bytes captured by the previous transform feedback to this buffer. If {@code pCounterBufferOffsets} is {@code NULL}, then it is assumed the offsets are zero.
     */
    public static void vkCmdBeginTransformFeedbackEXT(VkCommandBuffer commandBuffer, @NativeType("uint32_t") int firstCounterBuffer, @NativeType("VkBuffer const *") @Nullable LongBuffer pCounterBuffers, @NativeType("VkDeviceSize const *") @Nullable LongBuffer pCounterBufferOffsets) {
        if (CHECKS) {
            checkSafe(pCounterBufferOffsets, remainingSafe(pCounterBuffers));
        }
        nvkCmdBeginTransformFeedbackEXT(commandBuffer, firstCounterBuffer, remainingSafe(pCounterBuffers), memAddressSafe(pCounterBuffers), memAddressSafe(pCounterBufferOffsets));
    }

    // --- [ vkCmdEndTransformFeedbackEXT ] ---

    /**
     * Unsafe version of: {@link #vkCmdEndTransformFeedbackEXT CmdEndTransformFeedbackEXT}
     *
     * @param counterBufferCount the size of the {@code pCounterBuffers} and {@code pCounterBufferOffsets} arrays.
     */
    public static void nvkCmdEndTransformFeedbackEXT(VkCommandBuffer commandBuffer, int firstCounterBuffer, int counterBufferCount, long pCounterBuffers, long pCounterBufferOffsets) {
        long __functionAddress = commandBuffer.getCapabilities().vkCmdEndTransformFeedbackEXT;
        if (CHECKS) {
            check(__functionAddress);
        }
        callPPPV(commandBuffer.address(), firstCounterBuffer, counterBufferCount, pCounterBuffers, pCounterBufferOffsets, __functionAddress);
    }

    /**
     * Make transform feedback inactive in the command buffer.
     * 
     * <h5>C Specification</h5>
     * 
     * <p>Transform feedback for specific transform feedback buffers is made inactive by calling:</p>
     * 
     * <pre><code>
     * void vkCmdEndTransformFeedbackEXT(
     *     VkCommandBuffer                             commandBuffer,
     *     uint32_t                                    firstCounterBuffer,
     *     uint32_t                                    counterBufferCount,
     *     const VkBuffer*                             pCounterBuffers,
     *     const VkDeviceSize*                         pCounterBufferOffsets);</code></pre>
     * 
     * <h5>Valid Usage</h5>
     * 
     * <ul>
     * <li>{@link VkPhysicalDeviceTransformFeedbackFeaturesEXT}{@code ::transformFeedback} <b>must</b> be enabled</li>
     * <li>Transform feedback <b>must</b> be active</li>
     * <li>{@code firstCounterBuffer} <b>must</b> be less than {@link VkPhysicalDeviceTransformFeedbackPropertiesEXT}{@code ::maxTransformFeedbackBuffers}</li>
     * <li>The sum of {@code firstCounterBuffer} and {@code counterBufferCount} <b>must</b> be less than or equal to {@link VkPhysicalDeviceTransformFeedbackPropertiesEXT}{@code ::maxTransformFeedbackBuffers}</li>
     * <li>If {@code counterBufferCount} is not 0, and {@code pCounterBuffers} is not {@code NULL}, {@code pCounterBuffers} <b>must</b> be a valid pointer to an array of {@code counterBufferCount} {@code VkBuffer} handles that are either valid or {@link VK10#VK_NULL_HANDLE NULL_HANDLE}</li>
     * <li>For each buffer handle in the array, if it is not {@link VK10#VK_NULL_HANDLE NULL_HANDLE} it <b>must</b> reference a buffer large enough to hold 4 bytes at the corresponding offset from the {@code pCounterBufferOffsets} array</li>
     * <li>If {@code pCounterBuffer} is {@code NULL}, then {@code pCounterBufferOffsets} <b>must</b> also be {@code NULL}</li>
     * <li>For each buffer handle in the {@code pCounterBuffers} array that is not {@link VK10#VK_NULL_HANDLE NULL_HANDLE} it <b>must</b> have been created with a {@code usage} value containing {@link #VK_BUFFER_USAGE_TRANSFORM_FEEDBACK_COUNTER_BUFFER_BIT_EXT BUFFER_USAGE_TRANSFORM_FEEDBACK_COUNTER_BUFFER_BIT_EXT}</li>
     * </ul>
     * 
     * <h5>Valid Usage (Implicit)</h5>
     * 
     * <ul>
     * <li>{@code commandBuffer} <b>must</b> be a valid {@code VkCommandBuffer} handle</li>
     * <li>If {@code counterBufferCount} is not 0, and {@code pCounterBufferOffsets} is not {@code NULL}, {@code pCounterBufferOffsets} <b>must</b> be a valid pointer to an array of {@code counterBufferCount} {@code VkDeviceSize} values</li>
     * <li>{@code commandBuffer} <b>must</b> be in the <a href="https://registry.khronos.org/vulkan/specs/latest/html/vkspec.html#commandbuffers-lifecycle">recording state</a></li>
     * <li>The {@code VkCommandPool} that {@code commandBuffer} was allocated from <b>must</b> support graphics operations</li>
     * <li>This command <b>must</b> only be called inside of a render pass instance</li>
     * <li>This command <b>must</b> only be called outside of a video coding scope</li>
     * <li>Both of {@code commandBuffer}, and the elements of {@code pCounterBuffers} that are valid handles of non-ignored parameters <b>must</b> have been created, allocated, or retrieved from the same {@code VkDevice}</li>
     * </ul>
     * 
     * <h5>Host Synchronization</h5>
     * 
     * <ul>
     * <li>Host access to {@code commandBuffer} <b>must</b> be externally synchronized</li>
     * <li>Host access to the {@code VkCommandPool} that {@code commandBuffer} was allocated from <b>must</b> be externally synchronized</li>
     * </ul>
     * 
     * <h5>Command Properties</h5>
     * 
     * <table class="lwjgl">
     * <thead><tr><th><a href="https://registry.khronos.org/vulkan/specs/latest/html/vkspec.html#VkCommandBufferLevel">Command Buffer Levels</a></th><th><a href="https://registry.khronos.org/vulkan/specs/latest/html/vkspec.html#vkCmdBeginRenderPass">Render Pass Scope</a></th><th><a href="https://registry.khronos.org/vulkan/specs/latest/html/vkspec.html#vkCmdBeginVideoCodingKHR">Video Coding Scope</a></th><th><a href="https://registry.khronos.org/vulkan/specs/latest/html/vkspec.html#VkQueueFlagBits">Supported Queue Types</a></th><th><a href="https://registry.khronos.org/vulkan/specs/latest/html/vkspec.html#fundamentals-queueoperation-command-types">Command Type</a></th></tr></thead>
     * <tbody><tr><td>Primary Secondary</td><td>Inside</td><td>Outside</td><td>Graphics</td><td>State</td></tr></tbody>
     * </table>
     *
     * @param commandBuffer         the command buffer into which the command is recorded.
     * @param firstCounterBuffer    the index of the first transform feedback buffer corresponding to {@code pCounterBuffers}[0] and {@code pCounterBufferOffsets}[0].
     * @param pCounterBuffers       {@code NULL} or a pointer to an array of {@code VkBuffer} handles to counter buffers. The counter buffers are used to record the current byte positions of each transform feedback buffer where the next vertex output data would be captured. This <b>can</b> be used by a subsequent {@link #vkCmdBeginTransformFeedbackEXT CmdBeginTransformFeedbackEXT} call to resume transform feedback capture from this position. It can also be used by {@link #vkCmdDrawIndirectByteCountEXT CmdDrawIndirectByteCountEXT} to determine the vertex count of the draw call.
     * @param pCounterBufferOffsets {@code NULL} or a pointer to an array of {@code VkDeviceSize} values specifying offsets within each of the {@code pCounterBuffers} where the counter values can be written. The location in each counter buffer at these offsets <b>must</b> be large enough to contain 4 bytes of data. The data stored at this location is the byte offset from the start of the transform feedback buffer binding where the next vertex data would be written. If {@code pCounterBufferOffsets} is {@code NULL}, then it is assumed the offsets are zero.
     */
    public static void vkCmdEndTransformFeedbackEXT(VkCommandBuffer commandBuffer, @NativeType("uint32_t") int firstCounterBuffer, @NativeType("VkBuffer const *") @Nullable LongBuffer pCounterBuffers, @NativeType("VkDeviceSize const *") @Nullable LongBuffer pCounterBufferOffsets) {
        if (CHECKS) {
            checkSafe(pCounterBufferOffsets, remainingSafe(pCounterBuffers));
        }
        nvkCmdEndTransformFeedbackEXT(commandBuffer, firstCounterBuffer, remainingSafe(pCounterBuffers), memAddressSafe(pCounterBuffers), memAddressSafe(pCounterBufferOffsets));
    }

    // --- [ vkCmdBeginQueryIndexedEXT ] ---

    /**
     * Begin an indexed query.
     * 
     * <h5>C Specification</h5>
     * 
     * <p>To begin an indexed query, call:</p>
     * 
     * <pre><code>
     * void vkCmdBeginQueryIndexedEXT(
     *     VkCommandBuffer                             commandBuffer,
     *     VkQueryPool                                 queryPool,
     *     uint32_t                                    query,
     *     VkQueryControlFlags                         flags,
     *     uint32_t                                    index);</code></pre>
     * 
     * <h5>Description</h5>
     * 
     * <p>The {@code vkCmdBeginQueryIndexedEXT} command operates the same as the {@link VK10#vkCmdBeginQuery CmdBeginQuery} command, except that it also accepts a query type specific {@code index} parameter.</p>
     * 
     * <p>This command defines an execution dependency between other query commands that reference the same query index.</p>
     * 
     * <p>The first <a href="https://registry.khronos.org/vulkan/specs/latest/html/vkspec.html#synchronization-dependencies-scopes">synchronization scope</a> includes all commands which reference the queries in {@code queryPool} indicated by {@code query} and {@code index} that occur earlier in <a href="https://registry.khronos.org/vulkan/specs/latest/html/vkspec.html#synchronization-submission-order">submission order</a>.</p>
     * 
     * <p>The second <a href="https://registry.khronos.org/vulkan/specs/latest/html/vkspec.html#synchronization-dependencies-scopes">synchronization scope</a> includes all commands which reference the queries in {@code queryPool} indicated by {@code query} and {@code index} that occur later in <a href="https://registry.khronos.org/vulkan/specs/latest/html/vkspec.html#synchronization-submission-order">submission order</a>.</p>
     * 
     * <p>The operation of this command happens after the first scope and happens before the second scope.</p>
     * 
     * <h5>Valid Usage</h5>
     * 
     * <ul>
     * <li>All queries used by the command <b>must</b> be <em>unavailable</em></li>
     * <li>The {@code queryType} used to create {@code queryPool} <b>must</b> not be {@link VK10#VK_QUERY_TYPE_TIMESTAMP QUERY_TYPE_TIMESTAMP}</li>
     * <li>The {@code queryType} used to create {@code queryPool} <b>must</b> not be {@link KHRAccelerationStructure#VK_QUERY_TYPE_ACCELERATION_STRUCTURE_COMPACTED_SIZE_KHR QUERY_TYPE_ACCELERATION_STRUCTURE_COMPACTED_SIZE_KHR} or {@link KHRAccelerationStructure#VK_QUERY_TYPE_ACCELERATION_STRUCTURE_SERIALIZATION_SIZE_KHR QUERY_TYPE_ACCELERATION_STRUCTURE_SERIALIZATION_SIZE_KHR}</li>
     * <li>The {@code queryType} used to create {@code queryPool} <b>must</b> not be {@link KHRRayTracingMaintenance1#VK_QUERY_TYPE_ACCELERATION_STRUCTURE_SIZE_KHR QUERY_TYPE_ACCELERATION_STRUCTURE_SIZE_KHR} or {@link KHRRayTracingMaintenance1#VK_QUERY_TYPE_ACCELERATION_STRUCTURE_SERIALIZATION_BOTTOM_LEVEL_POINTERS_KHR QUERY_TYPE_ACCELERATION_STRUCTURE_SERIALIZATION_BOTTOM_LEVEL_POINTERS_KHR}</li>
     * <li>The {@code queryType} used to create {@code queryPool} <b>must</b> not be {@link NVRayTracing#VK_QUERY_TYPE_ACCELERATION_STRUCTURE_COMPACTED_SIZE_NV QUERY_TYPE_ACCELERATION_STRUCTURE_COMPACTED_SIZE_NV}</li>
     * <li>If the <a href="https://registry.khronos.org/vulkan/specs/latest/html/vkspec.html#features-occlusionQueryPrecise">{@code occlusionQueryPrecise}</a> feature is not enabled, or the {@code queryType} used to create {@code queryPool} was not {@link VK10#VK_QUERY_TYPE_OCCLUSION QUERY_TYPE_OCCLUSION}, {@code flags} <b>must</b> not contain {@link VK10#VK_QUERY_CONTROL_PRECISE_BIT QUERY_CONTROL_PRECISE_BIT}</li>
     * <li>{@code query} <b>must</b> be less than the number of queries in {@code queryPool}</li>
     * <li>If the {@code queryType} used to create {@code queryPool} was {@link VK10#VK_QUERY_TYPE_OCCLUSION QUERY_TYPE_OCCLUSION}, the {@code VkCommandPool} that {@code commandBuffer} was allocated from <b>must</b> support graphics operations</li>
     * <li>If the {@code queryType} used to create {@code queryPool} was {@link VK10#VK_QUERY_TYPE_PIPELINE_STATISTICS QUERY_TYPE_PIPELINE_STATISTICS} and any of the {@code pipelineStatistics} indicate graphics operations, the {@code VkCommandPool} that {@code commandBuffer} was allocated from <b>must</b> support graphics operations</li>
     * <li>If the {@code queryType} used to create {@code queryPool} was {@link VK10#VK_QUERY_TYPE_PIPELINE_STATISTICS QUERY_TYPE_PIPELINE_STATISTICS} and any of the {@code pipelineStatistics} indicate compute operations, the {@code VkCommandPool} that {@code commandBuffer} was allocated from <b>must</b> support compute operations</li>
     * <li>{@code commandBuffer} <b>must</b> not be a protected command buffer</li>
     * <li>If called within a render pass instance, the sum of {@code query} and the number of bits set in the current subpass’s view mask <b>must</b> be less than or equal to the number of queries in {@code queryPool}</li>
     * <li>If the {@code queryType} used to create {@code queryPool} was {@link KHRVideoQueue#VK_QUERY_TYPE_RESULT_STATUS_ONLY_KHR QUERY_TYPE_RESULT_STATUS_ONLY_KHR}, then the {@code VkCommandPool} that {@code commandBuffer} was allocated from <b>must</b> have been created with a queue family index that supports <a href="https://registry.khronos.org/vulkan/specs/latest/html/vkspec.html#queries-result-status-only">result status queries</a>, as indicated by {@link VkQueueFamilyQueryResultStatusPropertiesKHR}{@code ::queryResultStatusSupport}</li>
     * <li>If there is a bound video session, then there <b>must</b> be no <a href="https://registry.khronos.org/vulkan/specs/latest/html/vkspec.html#queries-operation-active">active</a> queries</li>
     * <li>If there is a bound video session, then it <b>must</b> not have been created with {@link KHRVideoMaintenance1#VK_VIDEO_SESSION_CREATE_INLINE_QUERIES_BIT_KHR VIDEO_SESSION_CREATE_INLINE_QUERIES_BIT_KHR}</li>
     * <li>If the {@code queryType} used to create {@code queryPool} was {@link KHRVideoQueue#VK_QUERY_TYPE_RESULT_STATUS_ONLY_KHR QUERY_TYPE_RESULT_STATUS_ONLY_KHR} and there is a bound video session, then {@code queryPool} <b>must</b> have been created with a {@link VkVideoProfileInfoKHR} structure included in the {@code pNext} chain of {@link VkQueryPoolCreateInfo} identical to the one specified in {@link VkVideoSessionCreateInfoKHR}{@code ::pVideoProfile} the bound video session was created with</li>
     * <li>If the {@code queryType} used to create {@code queryPool} was {@link KHRVideoEncodeQueue#VK_QUERY_TYPE_VIDEO_ENCODE_FEEDBACK_KHR QUERY_TYPE_VIDEO_ENCODE_FEEDBACK_KHR}, then there <b>must</b> be a bound video session</li>
     * <li>If the {@code queryType} used to create {@code queryPool} was {@link KHRVideoEncodeQueue#VK_QUERY_TYPE_VIDEO_ENCODE_FEEDBACK_KHR QUERY_TYPE_VIDEO_ENCODE_FEEDBACK_KHR} and there is a bound video session, then {@code queryPool} <b>must</b> have been created with a {@link VkVideoProfileInfoKHR} structure included in the {@code pNext} chain of {@link VkQueryPoolCreateInfo} identical to the one specified in {@link VkVideoSessionCreateInfoKHR}{@code ::pVideoProfile} the bound video session was created with</li>
     * <li>If the {@code queryType} used to create {@code queryPool} was not {@link KHRVideoQueue#VK_QUERY_TYPE_RESULT_STATUS_ONLY_KHR QUERY_TYPE_RESULT_STATUS_ONLY_KHR} or {@link KHRVideoEncodeQueue#VK_QUERY_TYPE_VIDEO_ENCODE_FEEDBACK_KHR QUERY_TYPE_VIDEO_ENCODE_FEEDBACK_KHR}, then there <b>must</b> be no bound video session</li>
     * <li>If the {@code queryPool} was created with the same {@code queryType} as that of another <a href="https://registry.khronos.org/vulkan/specs/latest/html/vkspec.html#queries-operation-active">active</a> query within {@code commandBuffer}, then {@code index} <b>must</b> not match the index used for the active query</li>
     * <li>If the {@code queryType} used to create {@code queryPool} was {@link #VK_QUERY_TYPE_TRANSFORM_FEEDBACK_STREAM_EXT QUERY_TYPE_TRANSFORM_FEEDBACK_STREAM_EXT} the {@code VkCommandPool} that {@code commandBuffer} was allocated from <b>must</b> support graphics operations</li>
     * <li>If the {@code queryType} used to create {@code queryPool} was {@link #VK_QUERY_TYPE_TRANSFORM_FEEDBACK_STREAM_EXT QUERY_TYPE_TRANSFORM_FEEDBACK_STREAM_EXT} the {@code index} parameter <b>must</b> be less than {@link VkPhysicalDeviceTransformFeedbackPropertiesEXT}{@code ::maxTransformFeedbackStreams}</li>
     * <li>If the {@code queryType} used to create {@code queryPool} was not {@link #VK_QUERY_TYPE_TRANSFORM_FEEDBACK_STREAM_EXT QUERY_TYPE_TRANSFORM_FEEDBACK_STREAM_EXT} and not {@link EXTPrimitivesGeneratedQuery#VK_QUERY_TYPE_PRIMITIVES_GENERATED_EXT QUERY_TYPE_PRIMITIVES_GENERATED_EXT}, the {@code index} <b>must</b> be zero</li>
     * <li>If the {@code queryType} used to create {@code queryPool} was {@link EXTPrimitivesGeneratedQuery#VK_QUERY_TYPE_PRIMITIVES_GENERATED_EXT QUERY_TYPE_PRIMITIVES_GENERATED_EXT} the {@code VkCommandPool} that {@code commandBuffer} was allocated from <b>must</b> support graphics operations</li>
     * <li>If the {@code queryType} used to create {@code queryPool} was {@link EXTPrimitivesGeneratedQuery#VK_QUERY_TYPE_PRIMITIVES_GENERATED_EXT QUERY_TYPE_PRIMITIVES_GENERATED_EXT} the {@code index} parameter <b>must</b> be less than {@link VkPhysicalDeviceTransformFeedbackPropertiesEXT}{@code ::maxTransformFeedbackStreams}</li>
     * <li>If the {@code queryType} used to create {@code queryPool} was {@link EXTPrimitivesGeneratedQuery#VK_QUERY_TYPE_PRIMITIVES_GENERATED_EXT QUERY_TYPE_PRIMITIVES_GENERATED_EXT} and the <a href="https://registry.khronos.org/vulkan/specs/latest/html/vkspec.html#features-primitivesGeneratedQueryWithNonZeroStreams">{@code primitivesGeneratedQueryWithNonZeroStreams}</a> feature is not enabled, the {@code index} parameter <b>must</b> be zero</li>
     * <li>If the {@code queryType} used to create {@code queryPool} was {@link EXTPrimitivesGeneratedQuery#VK_QUERY_TYPE_PRIMITIVES_GENERATED_EXT QUERY_TYPE_PRIMITIVES_GENERATED_EXT} then <a href="https://registry.khronos.org/vulkan/specs/latest/html/vkspec.html#features-primitivesGeneratedQuery">{@code primitivesGeneratedQuery}</a> <b>must</b> be enabled</li>
     * <li>If the {@code queryType} used to create {@code queryPool} was {@link #VK_QUERY_TYPE_TRANSFORM_FEEDBACK_STREAM_EXT QUERY_TYPE_TRANSFORM_FEEDBACK_STREAM_EXT} then {@link VkPhysicalDeviceTransformFeedbackPropertiesEXT}{@code ::transformFeedbackQueries} <b>must</b> be supported</li>
     * <li>The {@code queryType} used to create {@code queryPool} <b>must</b> not be {@link EXTMeshShader#VK_QUERY_TYPE_MESH_PRIMITIVES_GENERATED_EXT QUERY_TYPE_MESH_PRIMITIVES_GENERATED_EXT}</li>
     * </ul>
     * 
     * <ul>
     * <li>If {@code queryPool} was created with a {@code queryType} of {@link KHRPerformanceQuery#VK_QUERY_TYPE_PERFORMANCE_QUERY_KHR QUERY_TYPE_PERFORMANCE_QUERY_KHR}, then the {@link VkQueryPoolPerformanceCreateInfoKHR}{@code ::queueFamilyIndex} {@code queryPool} was created with <b>must</b> equal the queue family index of the {@code VkCommandPool} that {@code commandBuffer} was allocated from</li>
     * <li>If {@code queryPool} was created with a {@code queryType} of {@link KHRPerformanceQuery#VK_QUERY_TYPE_PERFORMANCE_QUERY_KHR QUERY_TYPE_PERFORMANCE_QUERY_KHR}, the <a href="https://registry.khronos.org/vulkan/specs/latest/html/vkspec.html#profiling-lock">profiling lock</a> <b>must</b> have been held before {@link VK10#vkBeginCommandBuffer BeginCommandBuffer} was called on {@code commandBuffer}</li>
     * <li>If {@code queryPool} was created with a {@code queryType} of {@link KHRPerformanceQuery#VK_QUERY_TYPE_PERFORMANCE_QUERY_KHR QUERY_TYPE_PERFORMANCE_QUERY_KHR} and one of the counters used to create {@code queryPool} was {@link KHRPerformanceQuery#VK_PERFORMANCE_COUNTER_SCOPE_COMMAND_BUFFER_KHR PERFORMANCE_COUNTER_SCOPE_COMMAND_BUFFER_KHR}, the query begin <b>must</b> be the first recorded command in {@code commandBuffer}</li>
     * <li>If {@code queryPool} was created with a {@code queryType} of {@link KHRPerformanceQuery#VK_QUERY_TYPE_PERFORMANCE_QUERY_KHR QUERY_TYPE_PERFORMANCE_QUERY_KHR} and one of the counters used to create {@code queryPool} was {@link KHRPerformanceQuery#VK_PERFORMANCE_COUNTER_SCOPE_RENDER_PASS_KHR PERFORMANCE_COUNTER_SCOPE_RENDER_PASS_KHR}, the begin command <b>must</b> not be recorded within a render pass instance</li>
     * <li>If {@code queryPool} was created with a {@code queryType} of {@link KHRPerformanceQuery#VK_QUERY_TYPE_PERFORMANCE_QUERY_KHR QUERY_TYPE_PERFORMANCE_QUERY_KHR} and another query pool with a {@code queryType} {@link KHRPerformanceQuery#VK_QUERY_TYPE_PERFORMANCE_QUERY_KHR QUERY_TYPE_PERFORMANCE_QUERY_KHR} has been used within {@code commandBuffer}, its parent primary command buffer or secondary command buffer recorded within the same parent primary command buffer as {@code commandBuffer}, the <a href="https://registry.khronos.org/vulkan/specs/latest/html/vkspec.html#features-performanceCounterMultipleQueryPools">{@code performanceCounterMultipleQueryPools}</a> feature <b>must</b> be enabled</li>
     * <li>If {@code queryPool} was created with a {@code queryType} of {@link KHRPerformanceQuery#VK_QUERY_TYPE_PERFORMANCE_QUERY_KHR QUERY_TYPE_PERFORMANCE_QUERY_KHR}, this command <b>must</b> not be recorded in a command buffer that, either directly or through secondary command buffers, also contains a {@code vkCmdResetQueryPool} command affecting the same query</li>
     * </ul>
     * 
     * <h5>Valid Usage (Implicit)</h5>
     * 
     * <ul>
     * <li>{@code commandBuffer} <b>must</b> be a valid {@code VkCommandBuffer} handle</li>
     * <li>{@code queryPool} <b>must</b> be a valid {@code VkQueryPool} handle</li>
     * <li>{@code flags} <b>must</b> be a valid combination of {@code VkQueryControlFlagBits} values</li>
     * <li>{@code commandBuffer} <b>must</b> be in the <a href="https://registry.khronos.org/vulkan/specs/latest/html/vkspec.html#commandbuffers-lifecycle">recording state</a></li>
     * <li>The {@code VkCommandPool} that {@code commandBuffer} was allocated from <b>must</b> support graphics, compute, decode, or encode operations</li>
     * <li>This command <b>must</b> only be called outside of a video coding scope</li>
     * <li>Both of {@code commandBuffer}, and {@code queryPool} <b>must</b> have been created, allocated, or retrieved from the same {@code VkDevice}</li>
     * </ul>
     * 
     * <h5>Host Synchronization</h5>
     * 
     * <ul>
     * <li>Host access to {@code commandBuffer} <b>must</b> be externally synchronized</li>
     * <li>Host access to the {@code VkCommandPool} that {@code commandBuffer} was allocated from <b>must</b> be externally synchronized</li>
     * </ul>
     * 
     * <h5>Command Properties</h5>
     * 
     * <table class="lwjgl">
     * <thead><tr><th><a href="https://registry.khronos.org/vulkan/specs/latest/html/vkspec.html#VkCommandBufferLevel">Command Buffer Levels</a></th><th><a href="https://registry.khronos.org/vulkan/specs/latest/html/vkspec.html#vkCmdBeginRenderPass">Render Pass Scope</a></th><th><a href="https://registry.khronos.org/vulkan/specs/latest/html/vkspec.html#vkCmdBeginVideoCodingKHR">Video Coding Scope</a></th><th><a href="https://registry.khronos.org/vulkan/specs/latest/html/vkspec.html#VkQueueFlagBits">Supported Queue Types</a></th><th><a href="https://registry.khronos.org/vulkan/specs/latest/html/vkspec.html#fundamentals-queueoperation-command-types">Command Type</a></th></tr></thead>
     * <tbody><tr><td>Primary Secondary</td><td>Both</td><td>Outside</td><td>Graphics Compute Decode Encode</td><td>Action State</td></tr></tbody>
     * </table>
     * 
     * <h5>See Also</h5>
     * 
     * <p>{@link VK10#vkCmdBeginQuery CmdBeginQuery}, {@link VK10#vkCmdEndQuery CmdEndQuery}, {@link #vkCmdEndQueryIndexedEXT CmdEndQueryIndexedEXT}</p>
     *
     * @param commandBuffer the command buffer into which this command will be recorded.
     * @param queryPool     the query pool that will manage the results of the query.
     * @param query         the query index within the query pool that will contain the results.
     * @param flags         a bitmask of {@code VkQueryControlFlagBits} specifying constraints on the types of queries that <b>can</b> be performed.
     * @param index         the query type specific index. When the query type is {@link #VK_QUERY_TYPE_TRANSFORM_FEEDBACK_STREAM_EXT QUERY_TYPE_TRANSFORM_FEEDBACK_STREAM_EXT} or {@link EXTPrimitivesGeneratedQuery#VK_QUERY_TYPE_PRIMITIVES_GENERATED_EXT QUERY_TYPE_PRIMITIVES_GENERATED_EXT}, the index represents the vertex stream.
     */
    public static void vkCmdBeginQueryIndexedEXT(VkCommandBuffer commandBuffer, @NativeType("VkQueryPool") long queryPool, @NativeType("uint32_t") int query, @NativeType("VkQueryControlFlags") int flags, @NativeType("uint32_t") int index) {
        long __functionAddress = commandBuffer.getCapabilities().vkCmdBeginQueryIndexedEXT;
        if (CHECKS) {
            check(__functionAddress);
        }
        callPJV(commandBuffer.address(), queryPool, query, flags, index, __functionAddress);
    }

    // --- [ vkCmdEndQueryIndexedEXT ] ---

    /**
     * Ends a query.
     * 
     * <h5>C Specification</h5>
     * 
     * <p>To end an indexed query after the set of desired drawing or dispatching commands is recorded, call:</p>
     * 
     * <pre><code>
     * void vkCmdEndQueryIndexedEXT(
     *     VkCommandBuffer                             commandBuffer,
     *     VkQueryPool                                 queryPool,
     *     uint32_t                                    query,
     *     uint32_t                                    index);</code></pre>
     * 
     * <h5>Description</h5>
     * 
     * <p>The command completes the query in {@code queryPool} identified by {@code query} and {@code index}, and marks it as available.</p>
     * 
     * <p>The {@code vkCmdEndQueryIndexedEXT} command operates the same as the {@link VK10#vkCmdEndQuery CmdEndQuery} command, except that it also accepts a query type specific {@code index} parameter.</p>
     * 
     * <p>This command defines an execution dependency between other query commands that reference the same query index.</p>
     * 
     * <p>The first <a href="https://registry.khronos.org/vulkan/specs/latest/html/vkspec.html#synchronization-dependencies-scopes">synchronization scope</a> includes all commands which reference the queries in {@code queryPool} indicated by {@code query} that occur earlier in <a href="https://registry.khronos.org/vulkan/specs/latest/html/vkspec.html#synchronization-submission-order">submission order</a>.</p>
     * 
     * <p>The second <a href="https://registry.khronos.org/vulkan/specs/latest/html/vkspec.html#synchronization-dependencies-scopes">synchronization scope</a> includes only the operation of this command.</p>
     * 
     * <h5>Valid Usage</h5>
     * 
     * <ul>
     * <li>All queries used by the command <b>must</b> be <a href="https://registry.khronos.org/vulkan/specs/latest/html/vkspec.html#queries-operation-active">active</a></li>
     * <li>{@code query} <b>must</b> be less than the number of queries in {@code queryPool}</li>
     * <li>{@code commandBuffer} <b>must</b> not be a protected command buffer</li>
     * <li>If {@code vkCmdEndQueryIndexedEXT} is called within a render pass instance, the sum of {@code query} and the number of bits set in the current subpass’s view mask <b>must</b> be less than or equal to the number of queries in {@code queryPool}</li>
     * <li>If the {@code queryType} used to create {@code queryPool} was {@link #VK_QUERY_TYPE_TRANSFORM_FEEDBACK_STREAM_EXT QUERY_TYPE_TRANSFORM_FEEDBACK_STREAM_EXT} or {@link EXTPrimitivesGeneratedQuery#VK_QUERY_TYPE_PRIMITIVES_GENERATED_EXT QUERY_TYPE_PRIMITIVES_GENERATED_EXT}, the {@code index} parameter <b>must</b> be less than {@link VkPhysicalDeviceTransformFeedbackPropertiesEXT}{@code ::maxTransformFeedbackStreams}</li>
     * <li>If the {@code queryType} used to create {@code queryPool} was not {@link #VK_QUERY_TYPE_TRANSFORM_FEEDBACK_STREAM_EXT QUERY_TYPE_TRANSFORM_FEEDBACK_STREAM_EXT} and not {@link EXTPrimitivesGeneratedQuery#VK_QUERY_TYPE_PRIMITIVES_GENERATED_EXT QUERY_TYPE_PRIMITIVES_GENERATED_EXT}, the {@code index} <b>must</b> be zero</li>
     * <li>If the {@code queryType} used to create {@code queryPool} was {@link #VK_QUERY_TYPE_TRANSFORM_FEEDBACK_STREAM_EXT QUERY_TYPE_TRANSFORM_FEEDBACK_STREAM_EXT} or {@link EXTPrimitivesGeneratedQuery#VK_QUERY_TYPE_PRIMITIVES_GENERATED_EXT QUERY_TYPE_PRIMITIVES_GENERATED_EXT}, {@code index} <b>must</b> equal the {@code index} used to begin the query</li>
     * </ul>
     * 
     * <ul>
     * <li>If called within a subpass of a render pass instance, the corresponding {@code vkCmdBeginQuery}* command <b>must</b> have been called previously within the same subpass</li>
     * </ul>
     * 
     * <h5>Valid Usage (Implicit)</h5>
     * 
     * <ul>
     * <li>{@code commandBuffer} <b>must</b> be a valid {@code VkCommandBuffer} handle</li>
     * <li>{@code queryPool} <b>must</b> be a valid {@code VkQueryPool} handle</li>
     * <li>{@code commandBuffer} <b>must</b> be in the <a href="https://registry.khronos.org/vulkan/specs/latest/html/vkspec.html#commandbuffers-lifecycle">recording state</a></li>
     * <li>The {@code VkCommandPool} that {@code commandBuffer} was allocated from <b>must</b> support graphics, compute, decode, or encode operations</li>
     * <li>This command <b>must</b> only be called outside of a video coding scope</li>
     * <li>Both of {@code commandBuffer}, and {@code queryPool} <b>must</b> have been created, allocated, or retrieved from the same {@code VkDevice}</li>
     * </ul>
     * 
     * <h5>Host Synchronization</h5>
     * 
     * <ul>
     * <li>Host access to {@code commandBuffer} <b>must</b> be externally synchronized</li>
     * <li>Host access to the {@code VkCommandPool} that {@code commandBuffer} was allocated from <b>must</b> be externally synchronized</li>
     * </ul>
     * 
     * <h5>Command Properties</h5>
     * 
     * <table class="lwjgl">
     * <thead><tr><th><a href="https://registry.khronos.org/vulkan/specs/latest/html/vkspec.html#VkCommandBufferLevel">Command Buffer Levels</a></th><th><a href="https://registry.khronos.org/vulkan/specs/latest/html/vkspec.html#vkCmdBeginRenderPass">Render Pass Scope</a></th><th><a href="https://registry.khronos.org/vulkan/specs/latest/html/vkspec.html#vkCmdBeginVideoCodingKHR">Video Coding Scope</a></th><th><a href="https://registry.khronos.org/vulkan/specs/latest/html/vkspec.html#VkQueueFlagBits">Supported Queue Types</a></th><th><a href="https://registry.khronos.org/vulkan/specs/latest/html/vkspec.html#fundamentals-queueoperation-command-types">Command Type</a></th></tr></thead>
     * <tbody><tr><td>Primary Secondary</td><td>Both</td><td>Outside</td><td>Graphics Compute Decode Encode</td><td>Action State</td></tr></tbody>
     * </table>
     * 
     * <h5>See Also</h5>
     * 
     * <p>{@link VK10#vkCmdBeginQuery CmdBeginQuery}, {@link #vkCmdBeginQueryIndexedEXT CmdBeginQueryIndexedEXT}, {@link VK10#vkCmdEndQuery CmdEndQuery}</p>
     *
     * @param commandBuffer the command buffer into which this command will be recorded.
     * @param queryPool     the query pool that is managing the results of the query.
     * @param query         the query index within the query pool where the result is stored.
     * @param index         the query type specific index.
     */
    public static void vkCmdEndQueryIndexedEXT(VkCommandBuffer commandBuffer, @NativeType("VkQueryPool") long queryPool, @NativeType("uint32_t") int query, @NativeType("uint32_t") int index) {
        long __functionAddress = commandBuffer.getCapabilities().vkCmdEndQueryIndexedEXT;
        if (CHECKS) {
            check(__functionAddress);
        }
        callPJV(commandBuffer.address(), queryPool, query, index, __functionAddress);
    }

    // --- [ vkCmdDrawIndirectByteCountEXT ] ---

    /**
     * Draw primitives with indirect parameters where the vertex count is derived from the counter byte value in the counter buffer.
     * 
     * <h5>C Specification</h5>
     * 
     * <p>To record a non-indexed draw call, where the vertex count is based on a byte count read from a buffer and the passed in vertex stride parameter, call:</p>
     * 
     * <pre><code>
     * void vkCmdDrawIndirectByteCountEXT(
     *     VkCommandBuffer                             commandBuffer,
     *     uint32_t                                    instanceCount,
     *     uint32_t                                    firstInstance,
     *     VkBuffer                                    counterBuffer,
     *     VkDeviceSize                                counterBufferOffset,
     *     uint32_t                                    counterOffset,
     *     uint32_t                                    vertexStride);</code></pre>
     * 
     * <h5>Description</h5>
     * 
     * <p>When the command is executed, primitives are assembled in the same way as done with {@link VK10#vkCmdDraw CmdDraw} except the {@code vertexCount} is calculated based on the byte count read from {@code counterBuffer} at offset {@code counterBufferOffset}. The assembled primitives execute the bound graphics pipeline.</p>
     * 
     * <p>The effective {@code vertexCount} is calculated as follows:</p>
     * 
     * <pre><code>
     * const uint32_t * counterBufferPtr = (const uint8_t *)counterBuffer.address + counterBufferOffset;
     * vertexCount = floor(max(0, (*counterBufferPtr - counterOffset)) / vertexStride);</code></pre>
     * 
     * <p>The effective {@code firstVertex} is zero.</p>
     * 
     * <h5>Valid Usage</h5>
     * 
     * <ul>
     * <li>If a {@code VkSampler} created with {@code magFilter} or {@code minFilter} equal to {@link VK10#VK_FILTER_LINEAR FILTER_LINEAR}, {@code reductionMode} equal to {@link VK12#VK_SAMPLER_REDUCTION_MODE_WEIGHTED_AVERAGE SAMPLER_REDUCTION_MODE_WEIGHTED_AVERAGE}, and {@code compareEnable} equal to {@link VK10#VK_FALSE FALSE} is used to sample a {@code VkImageView} as a result of this command, then the image view’s <a href="https://registry.khronos.org/vulkan/specs/latest/html/vkspec.html#resources-image-view-format-features">format features</a> <b>must</b> contain {@link VK10#VK_FORMAT_FEATURE_SAMPLED_IMAGE_FILTER_LINEAR_BIT FORMAT_FEATURE_SAMPLED_IMAGE_FILTER_LINEAR_BIT}</li>
     * <li>If a {@code VkSampler} created with {@code magFilter} or {@code minFilter} equal to {@link VK10#VK_FILTER_LINEAR FILTER_LINEAR} and {@code reductionMode} equal to either {@link VK12#VK_SAMPLER_REDUCTION_MODE_MIN SAMPLER_REDUCTION_MODE_MIN} or {@link VK12#VK_SAMPLER_REDUCTION_MODE_MAX SAMPLER_REDUCTION_MODE_MAX} is used to sample a {@code VkImageView} as a result of this command, then the image view’s <a href="https://registry.khronos.org/vulkan/specs/latest/html/vkspec.html#resources-image-view-format-features">format features</a> <b>must</b> contain {@link VK12#VK_FORMAT_FEATURE_SAMPLED_IMAGE_FILTER_MINMAX_BIT FORMAT_FEATURE_SAMPLED_IMAGE_FILTER_MINMAX_BIT}</li>
     * <li>If a {@code VkSampler} created with {@code mipmapMode} equal to {@link VK10#VK_SAMPLER_MIPMAP_MODE_LINEAR SAMPLER_MIPMAP_MODE_LINEAR}, {@code reductionMode} equal to {@link VK12#VK_SAMPLER_REDUCTION_MODE_WEIGHTED_AVERAGE SAMPLER_REDUCTION_MODE_WEIGHTED_AVERAGE}, and {@code compareEnable} equal to {@link VK10#VK_FALSE FALSE} is used to sample a {@code VkImageView} as a result of this command, then the image view’s <a href="https://registry.khronos.org/vulkan/specs/latest/html/vkspec.html#resources-image-view-format-features">format features</a> <b>must</b> contain {@link VK10#VK_FORMAT_FEATURE_SAMPLED_IMAGE_FILTER_LINEAR_BIT FORMAT_FEATURE_SAMPLED_IMAGE_FILTER_LINEAR_BIT}</li>
     * <li>If a {@code VkSampler} created with {@code mipmapMode} equal to {@link VK10#VK_SAMPLER_MIPMAP_MODE_LINEAR SAMPLER_MIPMAP_MODE_LINEAR} and {@code reductionMode} equal to either {@link VK12#VK_SAMPLER_REDUCTION_MODE_MIN SAMPLER_REDUCTION_MODE_MIN} or {@link VK12#VK_SAMPLER_REDUCTION_MODE_MAX SAMPLER_REDUCTION_MODE_MAX} is used to sample a {@code VkImageView} as a result of this command, then the image view’s <a href="https://registry.khronos.org/vulkan/specs/latest/html/vkspec.html#resources-image-view-format-features">format features</a> <b>must</b> contain {@link VK12#VK_FORMAT_FEATURE_SAMPLED_IMAGE_FILTER_MINMAX_BIT FORMAT_FEATURE_SAMPLED_IMAGE_FILTER_MINMAX_BIT}</li>
     * <li>If a {@code VkSampler} created with {@code unnormalizedCoordinates} equal to {@link VK10#VK_TRUE TRUE} is used to sample a {@code VkImageView} as a result of this command, then the image view’s {@code levelCount} and {@code layerCount} <b>must</b> be 1</li>
     * <li>If a {@code VkSampler} created with {@code unnormalizedCoordinates} equal to {@link VK10#VK_TRUE TRUE} is used to sample a {@code VkImageView} as a result of this command, then the image view’s {@code viewType} <b>must</b> be {@link VK10#VK_IMAGE_VIEW_TYPE_1D IMAGE_VIEW_TYPE_1D} or {@link VK10#VK_IMAGE_VIEW_TYPE_2D IMAGE_VIEW_TYPE_2D}</li>
     * <li>If a {@code VkSampler} created with {@code unnormalizedCoordinates} equal to {@link VK10#VK_TRUE TRUE} is used to sample a {@code VkImageView} as a result of this command, then the sampler <b>must</b> not be used with any of the SPIR-V {@code OpImageSample*} or {@code OpImageSparseSample*} instructions with {@code ImplicitLod}, {@code Dref} or {@code Proj} in their name</li>
     * <li>If a {@code VkSampler} created with {@code unnormalizedCoordinates} equal to {@link VK10#VK_TRUE TRUE} is used to sample a {@code VkImageView} as a result of this command, then the sampler <b>must</b> not be used with any of the SPIR-V {@code OpImageSample*} or {@code OpImageSparseSample*} instructions that includes a LOD bias or any offset values</li>
     * <li>If a {@code VkImageView} is sampled with <a href="https://registry.khronos.org/vulkan/specs/latest/html/vkspec.html#textures-depth-compare-operation">depth comparison</a>, the image view’s <a href="https://registry.khronos.org/vulkan/specs/latest/html/vkspec.html#resources-image-view-format-features">format features</a> <b>must</b> contain {@link VK13#VK_FORMAT_FEATURE_2_SAMPLED_IMAGE_DEPTH_COMPARISON_BIT FORMAT_FEATURE_2_SAMPLED_IMAGE_DEPTH_COMPARISON_BIT}</li>
     * <li>If a {@code VkImageView} is accessed using atomic operations as a result of this command, then the image view’s <a href="https://registry.khronos.org/vulkan/specs/latest/html/vkspec.html#resources-image-view-format-features">format features</a> <b>must</b> contain {@link VK10#VK_FORMAT_FEATURE_STORAGE_IMAGE_ATOMIC_BIT FORMAT_FEATURE_STORAGE_IMAGE_ATOMIC_BIT}</li>
     * <li>If a {@link VK10#VK_DESCRIPTOR_TYPE_STORAGE_TEXEL_BUFFER DESCRIPTOR_TYPE_STORAGE_TEXEL_BUFFER} descriptor is accessed using atomic operations as a result of this command, then the storage texel buffer’s <a href="https://registry.khronos.org/vulkan/specs/latest/html/vkspec.html#resources-buffer-view-format-features">format features</a> <b>must</b> contain {@link VK10#VK_FORMAT_FEATURE_STORAGE_TEXEL_BUFFER_ATOMIC_BIT FORMAT_FEATURE_STORAGE_TEXEL_BUFFER_ATOMIC_BIT}</li>
     * <li>If a {@code VkImageView} is sampled with {@link EXTFilterCubic#VK_FILTER_CUBIC_EXT FILTER_CUBIC_EXT} as a result of this command, then the image view’s <a href="https://registry.khronos.org/vulkan/specs/latest/html/vkspec.html#resources-image-view-format-features">format features</a> <b>must</b> contain {@link EXTFilterCubic#VK_FORMAT_FEATURE_SAMPLED_IMAGE_FILTER_CUBIC_BIT_EXT FORMAT_FEATURE_SAMPLED_IMAGE_FILTER_CUBIC_BIT_EXT}</li>
     * <li>If the {@link EXTFilterCubic VK_EXT_filter_cubic} extension is not enabled and any {@code VkImageView} is sampled with {@link EXTFilterCubic#VK_FILTER_CUBIC_EXT FILTER_CUBIC_EXT} as a result of this command, it <b>must</b> not have a {@code VkImageViewType} of {@link VK10#VK_IMAGE_VIEW_TYPE_3D IMAGE_VIEW_TYPE_3D}, {@link VK10#VK_IMAGE_VIEW_TYPE_CUBE IMAGE_VIEW_TYPE_CUBE}, or {@link VK10#VK_IMAGE_VIEW_TYPE_CUBE_ARRAY IMAGE_VIEW_TYPE_CUBE_ARRAY}</li>
     * <li>Any {@code VkImageView} being sampled with {@link EXTFilterCubic#VK_FILTER_CUBIC_EXT FILTER_CUBIC_EXT} as a result of this command <b>must</b> have a {@code VkImageViewType} and format that supports cubic filtering, as specified by {@link VkFilterCubicImageViewImageFormatPropertiesEXT}{@code ::filterCubic} returned by {@link VK11#vkGetPhysicalDeviceImageFormatProperties2 GetPhysicalDeviceImageFormatProperties2}</li>
     * <li>Any {@code VkImageView} being sampled with {@link EXTFilterCubic#VK_FILTER_CUBIC_EXT FILTER_CUBIC_EXT} with a reduction mode of either {@link VK12#VK_SAMPLER_REDUCTION_MODE_MIN SAMPLER_REDUCTION_MODE_MIN} or {@link VK12#VK_SAMPLER_REDUCTION_MODE_MAX SAMPLER_REDUCTION_MODE_MAX} as a result of this command <b>must</b> have a {@code VkImageViewType} and format that supports cubic filtering together with minmax filtering, as specified by {@link VkFilterCubicImageViewImageFormatPropertiesEXT}{@code ::filterCubicMinmax} returned by {@link VK11#vkGetPhysicalDeviceImageFormatProperties2 GetPhysicalDeviceImageFormatProperties2}</li>
     * <li>If the <a href="https://registry.khronos.org/vulkan/specs/latest/html/vkspec.html#features-cubicRangeClamp">{@code cubicRangeClamp}</a> feature is not enabled, then any {@code VkImageView} being sampled with {@link EXTFilterCubic#VK_FILTER_CUBIC_EXT FILTER_CUBIC_EXT} as a result of this command <b>must</b> not have a {@link VkSamplerReductionModeCreateInfo}{@code ::reductionMode} equal to {@link QCOMFilterCubicClamp#VK_SAMPLER_REDUCTION_MODE_WEIGHTED_AVERAGE_RANGECLAMP_QCOM SAMPLER_REDUCTION_MODE_WEIGHTED_AVERAGE_RANGECLAMP_QCOM}</li>
     * <li>Any {@code VkImageView} being sampled with a {@link VkSamplerReductionModeCreateInfo}{@code ::reductionMode} equal to {@link QCOMFilterCubicClamp#VK_SAMPLER_REDUCTION_MODE_WEIGHTED_AVERAGE_RANGECLAMP_QCOM SAMPLER_REDUCTION_MODE_WEIGHTED_AVERAGE_RANGECLAMP_QCOM} as a result of this command <b>must</b> sample with {@link EXTFilterCubic#VK_FILTER_CUBIC_EXT FILTER_CUBIC_EXT}</li>
     * <li>If the <a href="https://registry.khronos.org/vulkan/specs/latest/html/vkspec.html#features-selectableCubicWeights">{@code selectableCubicWeights}</a> feature is not enabled, then any {@code VkImageView} being sampled with {@link EXTFilterCubic#VK_FILTER_CUBIC_EXT FILTER_CUBIC_EXT} as a result of this command <b>must</b> have {@link VkSamplerCubicWeightsCreateInfoQCOM}{@code ::cubicWeights} equal to {@link QCOMFilterCubicWeights#VK_CUBIC_FILTER_WEIGHTS_CATMULL_ROM_QCOM CUBIC_FILTER_WEIGHTS_CATMULL_ROM_QCOM}</li>
     * <li>Any {@code VkImage} created with a {@link VkImageCreateInfo}{@code ::flags} containing {@link NVCornerSampledImage#VK_IMAGE_CREATE_CORNER_SAMPLED_BIT_NV IMAGE_CREATE_CORNER_SAMPLED_BIT_NV} sampled as a result of this command <b>must</b> only be sampled using a {@code VkSamplerAddressMode} of {@link VK10#VK_SAMPLER_ADDRESS_MODE_CLAMP_TO_EDGE SAMPLER_ADDRESS_MODE_CLAMP_TO_EDGE}</li>
     * <li>For any {@code VkImageView} being written as a storage image where the image format field of the {@code OpTypeImage} is {@code Unknown}, the view’s <a href="https://registry.khronos.org/vulkan/specs/latest/html/vkspec.html#resources-image-view-format-features">format features</a> <b>must</b> contain {@link VK13#VK_FORMAT_FEATURE_2_STORAGE_WRITE_WITHOUT_FORMAT_BIT FORMAT_FEATURE_2_STORAGE_WRITE_WITHOUT_FORMAT_BIT}</li>
     * <li>For any {@code VkImageView} being read as a storage image where the image format field of the {@code OpTypeImage} is {@code Unknown}, the view’s <a href="https://registry.khronos.org/vulkan/specs/latest/html/vkspec.html#resources-image-view-format-features">format features</a> <b>must</b> contain {@link VK13#VK_FORMAT_FEATURE_2_STORAGE_READ_WITHOUT_FORMAT_BIT FORMAT_FEATURE_2_STORAGE_READ_WITHOUT_FORMAT_BIT}</li>
     * <li>For any {@code VkBufferView} being written as a storage texel buffer where the image format field of the {@code OpTypeImage} is {@code Unknown}, the view’s <a href="https://registry.khronos.org/vulkan/specs/latest/html/vkspec.html#VkFormatProperties3">buffer features</a> <b>must</b> contain {@link VK13#VK_FORMAT_FEATURE_2_STORAGE_WRITE_WITHOUT_FORMAT_BIT FORMAT_FEATURE_2_STORAGE_WRITE_WITHOUT_FORMAT_BIT}</li>
     * <li>Any {@code VkBufferView} being read as a storage texel buffer where the image format field of the {@code OpTypeImage} is {@code Unknown} then the view’s <a href="https://registry.khronos.org/vulkan/specs/latest/html/vkspec.html#VkFormatProperties3">buffer features</a> <b>must</b> contain {@link VK13#VK_FORMAT_FEATURE_2_STORAGE_READ_WITHOUT_FORMAT_BIT FORMAT_FEATURE_2_STORAGE_READ_WITHOUT_FORMAT_BIT}</li>
     * <li>For each set <em>n</em> that is statically used by <a href="https://registry.khronos.org/vulkan/specs/latest/html/vkspec.html#shaders-binding">a bound shader</a>, a descriptor set <b>must</b> have been bound to <em>n</em> at the same pipeline bind point, with a {@code VkPipelineLayout} that is compatible for set <em>n</em>, with the {@code VkPipelineLayout} used to create the current {@code VkPipeline} or the {@code VkDescriptorSetLayout} array used to create the current {@code VkShaderEXT} , as described in <a href="https://registry.khronos.org/vulkan/specs/latest/html/vkspec.html#descriptorsets-compatibility">Pipeline Layout Compatibility</a></li>
     * <li>For each push constant that is statically used by <a href="https://registry.khronos.org/vulkan/specs/latest/html/vkspec.html#shaders-binding">a bound shader</a>, a push constant value <b>must</b> have been set for the same pipeline bind point, with a {@code VkPipelineLayout} that is compatible for push constants, with the {@code VkPipelineLayout} used to create the current {@code VkPipeline} or the {@code VkDescriptorSetLayout} array used to create the current {@code VkShaderEXT} , as described in <a href="https://registry.khronos.org/vulkan/specs/latest/html/vkspec.html#descriptorsets-compatibility">Pipeline Layout Compatibility</a></li>
     * <li>For each array of resources that is used by <a href="https://registry.khronos.org/vulkan/specs/latest/html/vkspec.html#shaders-binding">a bound shader</a>, the indices used to access members of the array <b>must</b> be less than the descriptor count for the identified binding in the descriptor sets used by this command</li>
     * <li>If the <a href="https://registry.khronos.org/vulkan/specs/latest/html/vkspec.html#features-maintenance4">{@code maintenance4}</a> feature is not enabled, then for each push constant that is statically used by <a href="https://registry.khronos.org/vulkan/specs/latest/html/vkspec.html#shaders-binding">a bound shader</a>, a push constant value <b>must</b> have been set for the same pipeline bind point, with a {@code VkPipelineLayout} that is compatible for push constants, with the {@code VkPipelineLayout} used to create the current {@code VkPipeline} or the {@code VkDescriptorSetLayout} and {@link VkPushConstantRange} arrays used to create the current {@code VkShaderEXT} , as described in <a href="https://registry.khronos.org/vulkan/specs/latest/html/vkspec.html#descriptorsets-compatibility">Pipeline Layout Compatibility</a></li>
     * <li>Descriptors in each bound descriptor set, specified via {@link VK10#vkCmdBindDescriptorSets CmdBindDescriptorSets}, <b>must</b> be valid as described by <a href="https://registry.khronos.org/vulkan/specs/latest/html/vkspec.html#descriptor-validity">descriptor validity</a> if they are statically used by the {@code VkPipeline} bound to the pipeline bind point used by this command and the bound {@code VkPipeline} was not created with {@link EXTDescriptorBuffer#VK_PIPELINE_CREATE_DESCRIPTOR_BUFFER_BIT_EXT PIPELINE_CREATE_DESCRIPTOR_BUFFER_BIT_EXT}</li>
     * <li>If the descriptors used by the {@code VkPipeline} bound to the pipeline bind point were specified via {@link VK10#vkCmdBindDescriptorSets CmdBindDescriptorSets}, the bound {@code VkPipeline} <b>must</b> have been created without {@link EXTDescriptorBuffer#VK_PIPELINE_CREATE_DESCRIPTOR_BUFFER_BIT_EXT PIPELINE_CREATE_DESCRIPTOR_BUFFER_BIT_EXT}</li>
     * <li>Descriptors in bound descriptor buffers, specified via {@link EXTDescriptorBuffer#vkCmdSetDescriptorBufferOffsetsEXT CmdSetDescriptorBufferOffsetsEXT}, <b>must</b> be valid if they are dynamically used by the {@code VkPipeline} bound to the pipeline bind point used by this command and the bound {@code VkPipeline} was created with {@link EXTDescriptorBuffer#VK_PIPELINE_CREATE_DESCRIPTOR_BUFFER_BIT_EXT PIPELINE_CREATE_DESCRIPTOR_BUFFER_BIT_EXT}</li>
     * <li>Descriptors in bound descriptor buffers, specified via {@link EXTDescriptorBuffer#vkCmdSetDescriptorBufferOffsetsEXT CmdSetDescriptorBufferOffsetsEXT}, <b>must</b> be valid if they are dynamically used by any {@code VkShaderEXT} bound to a stage corresponding to the pipeline bind point used by this command</li>
     * <li>If the descriptors used by the {@code VkPipeline} bound to the pipeline bind point were specified via {@link EXTDescriptorBuffer#vkCmdSetDescriptorBufferOffsetsEXT CmdSetDescriptorBufferOffsetsEXT}, the bound {@code VkPipeline} <b>must</b> have been created with {@link EXTDescriptorBuffer#VK_PIPELINE_CREATE_DESCRIPTOR_BUFFER_BIT_EXT PIPELINE_CREATE_DESCRIPTOR_BUFFER_BIT_EXT}</li>
     * <li>If a descriptor is dynamically used with a {@code VkPipeline} created with {@link EXTDescriptorBuffer#VK_PIPELINE_CREATE_DESCRIPTOR_BUFFER_BIT_EXT PIPELINE_CREATE_DESCRIPTOR_BUFFER_BIT_EXT}, the descriptor memory <b>must</b> be resident</li>
     * <li>If a descriptor is dynamically used with a {@code VkShaderEXT} created with a {@code VkDescriptorSetLayout} that was created with {@link EXTDescriptorBuffer#VK_DESCRIPTOR_SET_LAYOUT_CREATE_DESCRIPTOR_BUFFER_BIT_EXT DESCRIPTOR_SET_LAYOUT_CREATE_DESCRIPTOR_BUFFER_BIT_EXT}, the descriptor memory <b>must</b> be resident</li>
     * <li>If the <a href="https://registry.khronos.org/vulkan/specs/latest/html/vkspec.html#features-shaderObject">{@code shaderObject}</a> feature is not enabled, a valid pipeline <b>must</b> be bound to the pipeline bind point used by this command</li>
     * <li>If a pipeline is bound to the pipeline bind point used by this command, there <b>must</b> not have been any calls to dynamic state setting commands for any state specified statically in the {@code VkPipeline} object bound to the pipeline bind point used by this command, since that pipeline was bound</li>
     * <li>If the <a href="https://registry.khronos.org/vulkan/specs/latest/html/vkspec.html#features-shaderObject">{@code shaderObject}</a> feature is enabled, either a valid pipeline <b>must</b> be bound to the pipeline bind point used by this command, or a valid combination of valid and {@link VK10#VK_NULL_HANDLE NULL_HANDLE} shader objects <b>must</b> be bound to every supported shader stage corresponding to the pipeline bind point used by this command</li>
     * <li>If any stage of the {@code VkPipeline} object bound to the pipeline bind point used by this command accesses a uniform buffer, and that stage was created without enabling either {@link VK14#VK_PIPELINE_ROBUSTNESS_BUFFER_BEHAVIOR_ROBUST_BUFFER_ACCESS PIPELINE_ROBUSTNESS_BUFFER_BEHAVIOR_ROBUST_BUFFER_ACCESS} or {@link VK14#VK_PIPELINE_ROBUSTNESS_BUFFER_BEHAVIOR_ROBUST_BUFFER_ACCESS_2 PIPELINE_ROBUSTNESS_BUFFER_BEHAVIOR_ROBUST_BUFFER_ACCESS_2} for {@code uniformBuffers}, and the <a href="https://registry.khronos.org/vulkan/specs/latest/html/vkspec.html#features-robustBufferAccess">{@code robustBufferAccess}</a> feature is not enabled, that stage <b>must</b> not access values outside of the range of the buffer as specified in the descriptor set bound to the same pipeline bind point</li>
     * <li>If the <a href="https://registry.khronos.org/vulkan/specs/latest/html/vkspec.html#features-robustBufferAccess">{@code robustBufferAccess}</a> feature is not enabled, and any {@code VkShaderEXT} bound to a stage corresponding to the pipeline bind point used by this command accesses a uniform buffer, it <b>must</b> not access values outside of the range of the buffer as specified in the descriptor set bound to the same pipeline bind point</li>
     * <li>If any stage of the {@code VkPipeline} object bound to the pipeline bind point used by this command accesses a storage buffer, and that stage was created without enabling either {@link VK14#VK_PIPELINE_ROBUSTNESS_BUFFER_BEHAVIOR_ROBUST_BUFFER_ACCESS PIPELINE_ROBUSTNESS_BUFFER_BEHAVIOR_ROBUST_BUFFER_ACCESS} or {@link VK14#VK_PIPELINE_ROBUSTNESS_BUFFER_BEHAVIOR_ROBUST_BUFFER_ACCESS_2 PIPELINE_ROBUSTNESS_BUFFER_BEHAVIOR_ROBUST_BUFFER_ACCESS_2} for {@code storageBuffers}, and the <a href="https://registry.khronos.org/vulkan/specs/latest/html/vkspec.html#features-robustBufferAccess">{@code robustBufferAccess}</a> feature is not enabled, that stage <b>must</b> not access values outside of the range of the buffer as specified in the descriptor set bound to the same pipeline bind point</li>
     * <li>If the <a href="https://registry.khronos.org/vulkan/specs/latest/html/vkspec.html#features-robustBufferAccess">{@code robustBufferAccess}</a> feature is not enabled, and any {@code VkShaderEXT} bound to a stage corresponding to the pipeline bind point used by this command accesses a storage buffer, it <b>must</b> not access values outside of the range of the buffer as specified in the descriptor set bound to the same pipeline bind point</li>
     * <li>If {@code commandBuffer} is an unprotected command buffer and <a href="https://registry.khronos.org/vulkan/specs/latest/html/vkspec.html#limits-protectedNoFault">{@code protectedNoFault}</a> is not supported, any resource accessed by <a href="https://registry.khronos.org/vulkan/specs/latest/html/vkspec.html#shaders-binding">bound shaders</a> <b>must</b> not be a protected resource</li>
     * <li>If <a href="https://registry.khronos.org/vulkan/specs/latest/html/vkspec.html#shaders-binding">a bound shader</a> accesses a {@code VkSampler} or {@code VkImageView} object that enables <a href="https://registry.khronos.org/vulkan/specs/latest/html/vkspec.html#samplers-YCbCr-conversion">sampler Y′C<sub>B</sub>C<sub>R</sub> conversion</a>, that object <b>must</b> only be used with {@code OpImageSample*} or {@code OpImageSparseSample*} instructions</li>
     * <li>If <a href="https://registry.khronos.org/vulkan/specs/latest/html/vkspec.html#shaders-binding">a bound shader</a> accesses a {@code VkSampler} or {@code VkImageView} object that enables <a href="https://registry.khronos.org/vulkan/specs/latest/html/vkspec.html#samplers-YCbCr-conversion">sampler Y′C<sub>B</sub>C<sub>R</sub> conversion</a>, that object <b>must</b> not use the {@code ConstOffset} and {@code Offset} operands</li>
     * <li>If a {@code VkImageView} is accessed as a result of this command, then the image view’s {@code viewType} <b>must</b> match the {@code Dim} operand of the {@code OpTypeImage} as described in <a href="https://registry.khronos.org/vulkan/specs/latest/html/vkspec.html#spirvenv-image-dimensions">Compatibility Between SPIR-V Image Dimensions and Vulkan ImageView Types</a></li>
     * <li>If a {@code VkImageView} is accessed as a result of this command, then the <a href="https://registry.khronos.org/vulkan/specs/latest/html/vkspec.html#formats-numericformat">numeric type</a> of the image view’s {@code format} and the {@code Sampled} {@code Type} operand of the {@code OpTypeImage} <b>must</b> match</li>
     * <li>If a {@code VkImageView} created with a format other than {@link VK14#VK_FORMAT_A8_UNORM FORMAT_A8_UNORM} is accessed using {@code OpImageWrite} as a result of this command, then the {@code Type} of the {@code Texel} operand of that instruction <b>must</b> have at least as many components as the image view’s format</li>
     * <li>If a {@code VkImageView} created with the format {@link VK14#VK_FORMAT_A8_UNORM FORMAT_A8_UNORM} is accessed using {@code OpImageWrite} as a result of this command, then the {@code Type} of the {@code Texel} operand of that instruction <b>must</b> have four components</li>
     * <li>If a {@code VkBufferView} is accessed using {@code OpImageWrite} as a result of this command, then the {@code Type} of the {@code Texel} operand of that instruction <b>must</b> have at least as many components as the buffer view’s format</li>
     * <li>If a {@code VkImageView} with a {@code VkFormat} that has a 64-bit component width is accessed as a result of this command, the {@code SampledType} of the {@code OpTypeImage} operand of that instruction <b>must</b> have a {@code Width} of 64</li>
     * <li>If a {@code VkImageView} with a {@code VkFormat} that has a component width less than 64-bit is accessed as a result of this command, the {@code SampledType} of the {@code OpTypeImage} operand of that instruction <b>must</b> have a {@code Width} of 32</li>
     * <li>If a {@code VkBufferView} with a {@code VkFormat} that has a 64-bit component width is accessed as a result of this command, the {@code SampledType} of the {@code OpTypeImage} operand of that instruction <b>must</b> have a {@code Width} of 64</li>
     * <li>If a {@code VkBufferView} with a {@code VkFormat} that has a component width less than 64-bit is accessed as a result of this command, the {@code SampledType} of the {@code OpTypeImage} operand of that instruction <b>must</b> have a {@code Width} of 32</li>
     * <li>If the <a href="https://registry.khronos.org/vulkan/specs/latest/html/vkspec.html#features-sparseImageInt64Atomics">{@code sparseImageInt64Atomics}</a> feature is not enabled, {@code VkImage} objects created with the {@link VK10#VK_IMAGE_CREATE_SPARSE_RESIDENCY_BIT IMAGE_CREATE_SPARSE_RESIDENCY_BIT} flag <b>must</b> not be accessed by atomic instructions through an {@code OpTypeImage} with a {@code SampledType} with a {@code Width} of 64 by this command</li>
     * <li>If the <a href="https://registry.khronos.org/vulkan/specs/latest/html/vkspec.html#features-sparseImageInt64Atomics">{@code sparseImageInt64Atomics}</a> feature is not enabled, {@code VkBuffer} objects created with the {@link VK10#VK_BUFFER_CREATE_SPARSE_RESIDENCY_BIT BUFFER_CREATE_SPARSE_RESIDENCY_BIT} flag <b>must</b> not be accessed by atomic instructions through an {@code OpTypeImage} with a {@code SampledType} with a {@code Width} of 64 by this command</li>
     * <li>If {@code OpImageWeightedSampleQCOM} is used to sample a {@code VkImageView} as a result of this command, then the image view’s <a href="https://registry.khronos.org/vulkan/specs/latest/html/vkspec.html#resources-image-view-format-features">format features</a> <b>must</b> contain {@link QCOMImageProcessing#VK_FORMAT_FEATURE_2_WEIGHT_SAMPLED_IMAGE_BIT_QCOM FORMAT_FEATURE_2_WEIGHT_SAMPLED_IMAGE_BIT_QCOM}</li>
     * <li>If {@code OpImageWeightedSampleQCOM} uses a {@code VkImageView} as a sample weight image as a result of this command, then the image view’s <a href="https://registry.khronos.org/vulkan/specs/latest/html/vkspec.html#resources-image-view-format-features">format features</a> <b>must</b> contain {@link QCOMImageProcessing#VK_FORMAT_FEATURE_2_WEIGHT_IMAGE_BIT_QCOM FORMAT_FEATURE_2_WEIGHT_IMAGE_BIT_QCOM}</li>
     * <li>If {@code OpImageBoxFilterQCOM} is used to sample a {@code VkImageView} as a result of this command, then the image view’s <a href="https://registry.khronos.org/vulkan/specs/latest/html/vkspec.html#resources-image-view-format-features">format features</a> <b>must</b> contain {@link QCOMImageProcessing#VK_FORMAT_FEATURE_2_BOX_FILTER_SAMPLED_BIT_QCOM FORMAT_FEATURE_2_BOX_FILTER_SAMPLED_BIT_QCOM}</li>
     * <li>If {@code OpImageBlockMatchSSDQCOM} is used to read from an {@code VkImageView} as a result of this command, then the image view’s <a href="https://registry.khronos.org/vulkan/specs/latest/html/vkspec.html#resources-image-view-format-features">format features</a> <b>must</b> contain {@link QCOMImageProcessing#VK_FORMAT_FEATURE_2_BLOCK_MATCHING_BIT_QCOM FORMAT_FEATURE_2_BLOCK_MATCHING_BIT_QCOM}</li>
     * <li>If {@code OpImageBlockMatchSADQCOM} is used to read from an {@code VkImageView} as a result of this command, then the image view’s <a href="https://registry.khronos.org/vulkan/specs/latest/html/vkspec.html#resources-image-view-format-features">format features</a> <b>must</b> contain {@link QCOMImageProcessing#VK_FORMAT_FEATURE_2_BLOCK_MATCHING_BIT_QCOM FORMAT_FEATURE_2_BLOCK_MATCHING_BIT_QCOM}</li>
     * <li>If {@code OpImageBlockMatchSADQCOM} or OpImageBlockMatchSSDQCOM is used to read from a reference image as result of this command, then the specified reference coordinates <b>must</b> not fail <a href="https://registry.khronos.org/vulkan/specs/latest/html/vkspec.html#textures-integer-coordinate-validation">integer texel coordinate validation</a></li>
     * <li>If {@code OpImageWeightedSampleQCOM}, {@code OpImageBoxFilterQCOM}, {@code OpImageBlockMatchWindowSSDQCOM}, {@code OpImageBlockMatchWindowSADQCOM}, {@code OpImageBlockMatchGatherSSDQCOM}, {@code OpImageBlockMatchGatherSADQCOM}, {@code OpImageBlockMatchSSDQCOM}, or {@code OpImageBlockMatchSADQCOM} uses a {@code VkSampler} as a result of this command, then the sampler <b>must</b> have been created with {@link QCOMImageProcessing#VK_SAMPLER_CREATE_IMAGE_PROCESSING_BIT_QCOM SAMPLER_CREATE_IMAGE_PROCESSING_BIT_QCOM}</li>
     * <li>If any command other than {@code OpImageWeightedSampleQCOM}, {@code OpImageBoxFilterQCOM}, {@code OpImageBlockMatchWindowSSDQCOM}, {@code OpImageBlockMatchWindowSADQCOM}, {@code OpImageBlockMatchGatherSSDQCOM}, {@code OpImageBlockMatchGatherSADQCOM}, {@code OpImageBlockMatchSSDQCOM}, or {@code OpImageBlockMatchSADQCOM} uses a {@code VkSampler} as a result of this command, then the sampler <b>must</b> not have been created with {@link QCOMImageProcessing#VK_SAMPLER_CREATE_IMAGE_PROCESSING_BIT_QCOM SAMPLER_CREATE_IMAGE_PROCESSING_BIT_QCOM}</li>
     * <li>If a {@code OpImageBlockMatchWindow*QCOM} or {@code OpImageBlockMatchGather*QCOM} instruction is used to read from an {@code VkImageView} as a result of this command, then the image view’s <a href="https://registry.khronos.org/vulkan/specs/latest/html/vkspec.html#resources-image-view-format-features">format features</a> <b>must</b> contain {@link QCOMImageProcessing#VK_FORMAT_FEATURE_2_BLOCK_MATCHING_BIT_QCOM FORMAT_FEATURE_2_BLOCK_MATCHING_BIT_QCOM}</li>
     * <li>If a {@code OpImageBlockMatchWindow*QCOM} or {@code OpImageBlockMatchGather*QCOM} instruction is used to read from an {@code VkImageView} as a result of this command, then the image view’s format <b>must</b> be a single-component format</li>
     * <li>If a {@code OpImageBlockMatchWindow*QCOM} or {@code OpImageBlockMatchGather*QCOM} read from a reference image as result of this command, then the specified reference coordinates <b>must</b> not fail <a href="https://registry.khronos.org/vulkan/specs/latest/html/vkspec.html#textures-integer-coordinate-validation">integer texel coordinate validation</a></li>
     * <li>Any shader invocation executed by this command <b>must</b> <a href="https://registry.khronos.org/vulkan/specs/latest/html/vkspec.html#shaders-termination">terminate</a></li>
     * <li>If a descriptor with type equal to any of {@link QCOMImageProcessing#VK_DESCRIPTOR_TYPE_SAMPLE_WEIGHT_IMAGE_QCOM DESCRIPTOR_TYPE_SAMPLE_WEIGHT_IMAGE_QCOM}, {@link QCOMImageProcessing#VK_DESCRIPTOR_TYPE_BLOCK_MATCH_IMAGE_QCOM DESCRIPTOR_TYPE_BLOCK_MATCH_IMAGE_QCOM}, {@link VK10#VK_DESCRIPTOR_TYPE_SAMPLED_IMAGE DESCRIPTOR_TYPE_SAMPLED_IMAGE}, {@link VK10#VK_DESCRIPTOR_TYPE_STORAGE_IMAGE DESCRIPTOR_TYPE_STORAGE_IMAGE}, or {@link VK10#VK_DESCRIPTOR_TYPE_INPUT_ATTACHMENT DESCRIPTOR_TYPE_INPUT_ATTACHMENT} is accessed as a result of this command, the image subresource identified by that descriptor <b>must</b> be in the image layout identified when the descriptor was written</li>
     * <li>The current render pass <b>must</b> be <a href="https://registry.khronos.org/vulkan/specs/latest/html/vkspec.html#renderpass-compatibility">compatible</a> with the {@code renderPass} member of the {@link VkGraphicsPipelineCreateInfo} structure specified when creating the {@code VkPipeline} bound to {@link VK10#VK_PIPELINE_BIND_POINT_GRAPHICS PIPELINE_BIND_POINT_GRAPHICS}</li>
     * <li>The subpass index of the current render pass <b>must</b> be equal to the {@code subpass} member of the {@link VkGraphicsPipelineCreateInfo} structure specified when creating the {@code VkPipeline} bound to {@link VK10#VK_PIPELINE_BIND_POINT_GRAPHICS PIPELINE_BIND_POINT_GRAPHICS}</li>
     * <li>If any shader statically accesses an input attachment, a valid descriptor <b>must</b> be bound to the pipeline via a descriptor set</li>
     * <li>If any shader executed by this pipeline accesses an {@code OpTypeImage} variable with a {@code Dim} operand of {@code SubpassData}, it <b>must</b> be decorated with an {@code InputAttachmentIndex} that corresponds to a valid input attachment in the current subpass</li>
     * <li>Input attachment views accessed in a subpass <b>must</b> be created with the same {@code VkFormat} as the corresponding subpass definition, and be created with a {@code VkImageView} that is compatible with the attachment referenced by the subpass' {@code pInputAttachments}[{@code InputAttachmentIndex}] in the bound {@code VkFramebuffer} as specified by <a href="https://registry.khronos.org/vulkan/specs/latest/html/vkspec.html#compatibility-inputattachment">Fragment Input Attachment Compatibility</a></li>
     * <li>Input attachment views accessed in a dynamic render pass with a {@code InputAttachmentIndex} referenced by {@link VkRenderingInputAttachmentIndexInfo}, or no {@code InputAttachmentIndex} if {@link VkRenderingInputAttachmentIndexInfo}:{@code pDepthInputAttachmentIndex} or {@link VkRenderingInputAttachmentIndexInfo}:{@code pStencilInputAttachmentIndex} are {@code NULL}, <b>must</b> be created with a {@code VkImageView} that is compatible with the corresponding color, depth, or stencil attachment in {@link VkRenderingInfo}</li>
     * <li>Input attachment views accessed in a dynamic render pass via a shader object <b>must</b> have an {@code InputAttachmentIndex} if both {@link VkRenderingInputAttachmentIndexInfo}:{@code pDepthInputAttachmentIndex} and {@link VkRenderingInputAttachmentIndexInfo}:{@code pStencilInputAttachmentIndex} are non-{@code NULL}</li>
     * <li>If an input attachment view accessed in a dynamic render pass via a shader object has an {@code InputAttachmentIndex}, the {@code InputAttachmentIndex} <b>must</b> match an index in {@link VkRenderingInputAttachmentIndexInfo}</li>
     * <li>Memory backing image subresources used as attachments in the current render pass <b>must</b> not be written in any way other than as an attachment by this command</li>
     * <li>If a color attachment is written by any prior command in this subpass or by the load, store, or resolve operations for this subpass, it is not in the {@link EXTAttachmentFeedbackLoopLayout#VK_IMAGE_LAYOUT_ATTACHMENT_FEEDBACK_LOOP_OPTIMAL_EXT IMAGE_LAYOUT_ATTACHMENT_FEEDBACK_LOOP_OPTIMAL_EXT} image layout, and either:
     * 
     * <ul>
     * <li>the {@link EXTAttachmentFeedbackLoopLayout#VK_PIPELINE_CREATE_COLOR_ATTACHMENT_FEEDBACK_LOOP_BIT_EXT PIPELINE_CREATE_COLOR_ATTACHMENT_FEEDBACK_LOOP_BIT_EXT} is set on the bound pipeline or</li>
     * <li>the last call to {@link EXTAttachmentFeedbackLoopDynamicState#vkCmdSetAttachmentFeedbackLoopEnableEXT CmdSetAttachmentFeedbackLoopEnableEXT} included {@link VK10#VK_IMAGE_ASPECT_COLOR_BIT IMAGE_ASPECT_COLOR_BIT} and
     * 
     * <ul>
     * <li>there is no bound graphics pipeline or</li>
     * <li>the bound graphics pipeline was created with {@link EXTAttachmentFeedbackLoopDynamicState#VK_DYNAMIC_STATE_ATTACHMENT_FEEDBACK_LOOP_ENABLE_EXT DYNAMIC_STATE_ATTACHMENT_FEEDBACK_LOOP_ENABLE_EXT}</li>
     * </ul>
     * </li>
     * </ul>
     * 
     * <p>it <b>must</b> not be accessed in any way other than as an attachment by this command</p>
     * </li>
     * <li>If a depth attachment is written by any prior command in this subpass or by the load, store, or resolve operations for this subpass, it is not in the {@link EXTAttachmentFeedbackLoopLayout#VK_IMAGE_LAYOUT_ATTACHMENT_FEEDBACK_LOOP_OPTIMAL_EXT IMAGE_LAYOUT_ATTACHMENT_FEEDBACK_LOOP_OPTIMAL_EXT} image layout, and either:
     * 
     * <ul>
     * <li>the {@link EXTAttachmentFeedbackLoopLayout#VK_PIPELINE_CREATE_DEPTH_STENCIL_ATTACHMENT_FEEDBACK_LOOP_BIT_EXT PIPELINE_CREATE_DEPTH_STENCIL_ATTACHMENT_FEEDBACK_LOOP_BIT_EXT} is set on the bound pipeline or</li>
     * <li>the last call to {@link EXTAttachmentFeedbackLoopDynamicState#vkCmdSetAttachmentFeedbackLoopEnableEXT CmdSetAttachmentFeedbackLoopEnableEXT} included {@link VK10#VK_IMAGE_ASPECT_DEPTH_BIT IMAGE_ASPECT_DEPTH_BIT} and
     * 
     * <ul>
     * <li>there is no bound graphics pipeline or</li>
     * <li>the bound graphics pipeline was created with {@link EXTAttachmentFeedbackLoopDynamicState#VK_DYNAMIC_STATE_ATTACHMENT_FEEDBACK_LOOP_ENABLE_EXT DYNAMIC_STATE_ATTACHMENT_FEEDBACK_LOOP_ENABLE_EXT}</li>
     * </ul>
     * </li>
     * </ul>
     * 
     * <p>it <b>must</b> not be accessed in any way other than as an attachment by this command</p>
     * </li>
     * <li>If a stencil attachment is written by any prior command in this subpass or by the load, store, or resolve operations for this subpass, it is not in the {@link EXTAttachmentFeedbackLoopLayout#VK_IMAGE_LAYOUT_ATTACHMENT_FEEDBACK_LOOP_OPTIMAL_EXT IMAGE_LAYOUT_ATTACHMENT_FEEDBACK_LOOP_OPTIMAL_EXT} image layout, and either:
     * 
     * <ul>
     * <li>the {@link EXTAttachmentFeedbackLoopLayout#VK_PIPELINE_CREATE_DEPTH_STENCIL_ATTACHMENT_FEEDBACK_LOOP_BIT_EXT PIPELINE_CREATE_DEPTH_STENCIL_ATTACHMENT_FEEDBACK_LOOP_BIT_EXT} is set on the bound pipeline or</li>
     * <li>the last call to {@link EXTAttachmentFeedbackLoopDynamicState#vkCmdSetAttachmentFeedbackLoopEnableEXT CmdSetAttachmentFeedbackLoopEnableEXT} included {@link VK10#VK_IMAGE_ASPECT_STENCIL_BIT IMAGE_ASPECT_STENCIL_BIT} and
     * 
     * <ul>
     * <li>there is no bound graphics pipeline or</li>
     * <li>the bound graphics pipeline was created with {@link EXTAttachmentFeedbackLoopDynamicState#VK_DYNAMIC_STATE_ATTACHMENT_FEEDBACK_LOOP_ENABLE_EXT DYNAMIC_STATE_ATTACHMENT_FEEDBACK_LOOP_ENABLE_EXT}</li>
     * </ul>
     * </li>
     * </ul>
     * 
     * <p>it <b>must</b> not be accessed in any way other than as an attachment by this command</p>
     * </li>
     * <li>If an attachment is written by any prior command in this subpass or by the load, store, or resolve operations for this subpass, it <b>must</b> not be accessed in any way other than as an attachment, storage image, or sampled image by this command</li>
     * <li>If any previously recorded command in the current subpass accessed an image subresource used as an attachment in this subpass in any way other than as an attachment, this command <b>must</b> not write to that image subresource as an attachment</li>
     * <li>If the current render pass instance uses a depth/stencil attachment with a read-only layout for the depth aspect, <a href="https://registry.khronos.org/vulkan/specs/latest/html/vkspec.html#fragops-depth-write">depth writes</a> <b>must</b> be disabled</li>
     * <li>If the current render pass instance uses a depth/stencil attachment with a read-only layout for the stencil aspect, both front and back {@code writeMask} are not zero, and stencil test is enabled, <a href="https://registry.khronos.org/vulkan/specs/latest/html/vkspec.html#fragops-stencil">all stencil ops</a> <b>must</b> be {@link VK10#VK_STENCIL_OP_KEEP STENCIL_OP_KEEP}</li>
     * <li>If the bound graphics pipeline state was created with the {@link VK10#VK_DYNAMIC_STATE_VIEWPORT DYNAMIC_STATE_VIEWPORT} dynamic state enabled then {@link VK10#vkCmdSetViewport CmdSetViewport} <b>must</b> have been called and not subsequently <a href="https://registry.khronos.org/vulkan/specs/latest/html/vkspec.html#dynamic-state-lifetime">invalidated</a> in the current command buffer prior to this drawing command</li>
     * <li>If the bound graphics pipeline state was created with the {@link VK10#VK_DYNAMIC_STATE_SCISSOR DYNAMIC_STATE_SCISSOR} dynamic state enabled then {@link VK10#vkCmdSetScissor CmdSetScissor} <b>must</b> have been called and not subsequently <a href="https://registry.khronos.org/vulkan/specs/latest/html/vkspec.html#dynamic-state-lifetime">invalidated</a> in the current command buffer prior to this drawing command</li>
     * <li>If the bound graphics pipeline state was created with the {@link VK10#VK_DYNAMIC_STATE_LINE_WIDTH DYNAMIC_STATE_LINE_WIDTH} dynamic state enabled then {@link VK10#vkCmdSetLineWidth CmdSetLineWidth} <b>must</b> have been called and not subsequently <a href="https://registry.khronos.org/vulkan/specs/latest/html/vkspec.html#dynamic-state-lifetime">invalidated</a> in the current command buffer prior to this drawing command</li>
     * <li>If a shader object is bound to any graphics stage, and the most recent call to {@link VK13#vkCmdSetRasterizerDiscardEnable CmdSetRasterizerDiscardEnable} in the current command buffer set {@code rasterizerDiscardEnable} to {@link VK10#VK_FALSE FALSE}, and the most recent call to {@link EXTShaderObject#vkCmdSetPolygonModeEXT CmdSetPolygonModeEXT} in the current command buffer set {@code polygonMode} to {@link VK10#VK_POLYGON_MODE_LINE POLYGON_MODE_LINE}, {@link VK10#vkCmdSetLineWidth CmdSetLineWidth} <b>must</b> have been called and not subsequently <a href="https://registry.khronos.org/vulkan/specs/latest/html/vkspec.html#dynamic-state-lifetime">invalidated</a> in the current command buffer prior to this drawing command</li>
     * <li>If a shader object is bound to any graphics stage, and the most recent call to {@link VK13#vkCmdSetRasterizerDiscardEnable CmdSetRasterizerDiscardEnable} in the current command buffer set {@code rasterizerDiscardEnable} to {@link VK10#VK_FALSE FALSE}, and the most recent call to {@link VK13#vkCmdSetPrimitiveTopology CmdSetPrimitiveTopology} in the current command buffer set {@code primitiveTopology} to any line topology, {@link VK10#vkCmdSetLineWidth CmdSetLineWidth} <b>must</b> have been called and not subsequently <a href="https://registry.khronos.org/vulkan/specs/latest/html/vkspec.html#dynamic-state-lifetime">invalidated</a> in the current command buffer prior to this drawing command</li>
     * <li>If a shader object that outputs line primitives is bound to the {@link VK10#VK_SHADER_STAGE_TESSELLATION_EVALUATION_BIT SHADER_STAGE_TESSELLATION_EVALUATION_BIT} or {@link VK10#VK_SHADER_STAGE_GEOMETRY_BIT SHADER_STAGE_GEOMETRY_BIT} stage, and the most recent call to {@link VK13#vkCmdSetRasterizerDiscardEnable CmdSetRasterizerDiscardEnable} in the current command buffer set {@code rasterizerDiscardEnable} to {@link VK10#VK_FALSE FALSE}, {@link VK10#vkCmdSetLineWidth CmdSetLineWidth} <b>must</b> have been called and not subsequently <a href="https://registry.khronos.org/vulkan/specs/latest/html/vkspec.html#dynamic-state-lifetime">invalidated</a> in the current command buffer prior to this drawing command</li>
     * <li>If a shader object is bound to any graphics stage or a graphics pipeline is bound which was created with the {@link VK10#VK_DYNAMIC_STATE_DEPTH_BIAS DYNAMIC_STATE_DEPTH_BIAS} dynamic state enabled, the <a href="https://registry.khronos.org/vulkan/specs/latest/html/vkspec.html#dynamic-state-current-value">current value</a> of {@code rasterizerDiscardEnable} is {@link VK10#VK_FALSE FALSE}, and the <a href="https://registry.khronos.org/vulkan/specs/latest/html/vkspec.html#dynamic-state-current-value">current value</a> of {@code depthBiasEnable} is {@link VK10#VK_TRUE TRUE}, then {@link VK10#vkCmdSetDepthBias CmdSetDepthBias} or {@link EXTDepthBiasControl#vkCmdSetDepthBias2EXT CmdSetDepthBias2EXT} <b>must</b> have been called and not subsequently <a href="https://registry.khronos.org/vulkan/specs/latest/html/vkspec.html#dynamic-state-lifetime">invalidated</a> in the current command buffer prior to this drawing command</li>
     * <li>If the bound graphics pipeline state was created with the {@link VK10#VK_DYNAMIC_STATE_BLEND_CONSTANTS DYNAMIC_STATE_BLEND_CONSTANTS} dynamic state enabled then {@link VK10#vkCmdSetBlendConstants CmdSetBlendConstants} <b>must</b> have been called and not subsequently <a href="https://registry.khronos.org/vulkan/specs/latest/html/vkspec.html#dynamic-state-lifetime">invalidated</a> in the current command buffer prior to this drawing command</li>
     * <li>If a shader object is bound to the {@link VK10#VK_SHADER_STAGE_FRAGMENT_BIT SHADER_STAGE_FRAGMENT_BIT} stage, and the most recent call to {@link VK13#vkCmdSetRasterizerDiscardEnable CmdSetRasterizerDiscardEnable} in the current command buffer set {@code rasterizerDiscardEnable} to {@link VK10#VK_FALSE FALSE}, and the most recent call to {@link EXTShaderObject#vkCmdSetColorBlendEnableEXT CmdSetColorBlendEnableEXT} in the current command buffer set any element of {@code pColorBlendEnables} to {@link VK10#VK_TRUE TRUE}, and the most recent call to {@link EXTShaderObject#vkCmdSetColorBlendEquationEXT CmdSetColorBlendEquationEXT} in the current command buffer set the same element of {@code pColorBlendEquations} to a {@link VkColorBlendEquationEXT} structure with any {@code VkBlendFactor} member with a value of {@link VK10#VK_BLEND_FACTOR_CONSTANT_COLOR BLEND_FACTOR_CONSTANT_COLOR}, {@link VK10#VK_BLEND_FACTOR_ONE_MINUS_CONSTANT_COLOR BLEND_FACTOR_ONE_MINUS_CONSTANT_COLOR}, {@link VK10#VK_BLEND_FACTOR_CONSTANT_ALPHA BLEND_FACTOR_CONSTANT_ALPHA}, or {@link VK10#VK_BLEND_FACTOR_ONE_MINUS_CONSTANT_ALPHA BLEND_FACTOR_ONE_MINUS_CONSTANT_ALPHA}, {@link VK10#vkCmdSetBlendConstants CmdSetBlendConstants} <b>must</b> have been called and not subsequently <a href="https://registry.khronos.org/vulkan/specs/latest/html/vkspec.html#dynamic-state-lifetime">invalidated</a> in the current command buffer prior to this drawing command</li>
     * <li>If a shader object is bound to any graphics stage or a graphics pipeline is bound which was created with the {@link VK10#VK_DYNAMIC_STATE_DEPTH_BOUNDS DYNAMIC_STATE_DEPTH_BOUNDS} dynamic state enabled, the <a href="https://registry.khronos.org/vulkan/specs/latest/html/vkspec.html#dynamic-state-current-value">current value</a> of {@code rasterizerDiscardEnable} is {@link VK10#VK_FALSE FALSE}, and the <a href="https://registry.khronos.org/vulkan/specs/latest/html/vkspec.html#dynamic-state-current-value">current value</a> of {@code depthBoundsTestEnable} is {@link VK10#VK_TRUE TRUE}, then {@link VK10#vkCmdSetDepthBounds CmdSetDepthBounds} <b>must</b> have been called and not subsequently <a href="https://registry.khronos.org/vulkan/specs/latest/html/vkspec.html#dynamic-state-lifetime">invalidated</a> in the current command buffer prior to this drawing command</li>
     * <li>If a shader object is bound to any graphics stage or a graphics pipeline is bound which was created with the {@link VK10#VK_DYNAMIC_STATE_STENCIL_COMPARE_MASK DYNAMIC_STATE_STENCIL_COMPARE_MASK} dynamic state enabled, the <a href="https://registry.khronos.org/vulkan/specs/latest/html/vkspec.html#dynamic-state-current-value">current value</a> of {@code rasterizerDiscardEnable} is {@link VK10#VK_FALSE FALSE}, and the <a href="https://registry.khronos.org/vulkan/specs/latest/html/vkspec.html#dynamic-state-current-value">current value</a> of {@code stencilTestEnable} is {@link VK10#VK_TRUE TRUE}, then {@link VK10#vkCmdSetStencilCompareMask CmdSetStencilCompareMask} <b>must</b> have been called and not subsequently <a href="https://registry.khronos.org/vulkan/specs/latest/html/vkspec.html#dynamic-state-lifetime">invalidated</a> in the current command buffer prior to this drawing command</li>
     * <li>If a shader object is bound to any graphics stage or a graphics pipeline is bound which was created with the {@link VK10#VK_DYNAMIC_STATE_STENCIL_WRITE_MASK DYNAMIC_STATE_STENCIL_WRITE_MASK} dynamic state enabled, the <a href="https://registry.khronos.org/vulkan/specs/latest/html/vkspec.html#dynamic-state-current-value">current value</a> of {@code rasterizerDiscardEnable} is {@link VK10#VK_FALSE FALSE}, and the <a href="https://registry.khronos.org/vulkan/specs/latest/html/vkspec.html#dynamic-state-current-value">current value</a> of {@code stencilTestEnable} is {@link VK10#VK_TRUE TRUE}, then {@link VK10#vkCmdSetStencilWriteMask CmdSetStencilWriteMask} <b>must</b> have been called and not subsequently <a href="https://registry.khronos.org/vulkan/specs/latest/html/vkspec.html#dynamic-state-lifetime">invalidated</a> in the current command buffer prior to this drawing command</li>
     * <li>If a shader object is bound to any graphics stage or a graphics pipeline is bound which was created with the {@link VK10#VK_DYNAMIC_STATE_STENCIL_REFERENCE DYNAMIC_STATE_STENCIL_REFERENCE} dynamic state enabled, the <a href="https://registry.khronos.org/vulkan/specs/latest/html/vkspec.html#dynamic-state-current-value">current value</a> of and {@code rasterizerDiscardEnable} is {@link VK10#VK_FALSE FALSE}, the <a href="https://registry.khronos.org/vulkan/specs/latest/html/vkspec.html#dynamic-state-current-value">current value</a> of {@code stencilTestEnable} is {@link VK10#VK_TRUE TRUE}, then {@link VK10#vkCmdSetStencilReference CmdSetStencilReference} <b>must</b> have been called and not subsequently <a href="https://registry.khronos.org/vulkan/specs/latest/html/vkspec.html#dynamic-state-lifetime">invalidated</a> in the current command buffer prior to this drawing command</li>
     * <li>If the draw is recorded in a render pass instance with multiview enabled, the maximum instance index <b>must</b> be less than or equal to {@link VkPhysicalDeviceMultiviewProperties}{@code ::maxMultiviewInstanceIndex}</li>
     * <li>If the bound graphics pipeline was created with {@link VkPipelineSampleLocationsStateCreateInfoEXT}{@code ::sampleLocationsEnable} set to {@link VK10#VK_TRUE TRUE} and the current subpass has a depth/stencil attachment, then that attachment <b>must</b> have been created with the {@link EXTSampleLocations#VK_IMAGE_CREATE_SAMPLE_LOCATIONS_COMPATIBLE_DEPTH_BIT_EXT IMAGE_CREATE_SAMPLE_LOCATIONS_COMPATIBLE_DEPTH_BIT_EXT} bit set</li>
     * <li>If the {@link EXTSampleLocations VK_EXT_sample_locations} extension is enabled, a shader object is bound to any graphics stage or a graphics pipeline is bound which was created with the {@link EXTExtendedDynamicState3#VK_DYNAMIC_STATE_SAMPLE_LOCATIONS_ENABLE_EXT DYNAMIC_STATE_SAMPLE_LOCATIONS_ENABLE_EXT} dynamic state enabled, and the <a href="https://registry.khronos.org/vulkan/specs/latest/html/vkspec.html#dynamic-state-current-value">current value</a> of {@code rasterizerDiscardEnable} is {@link VK10#VK_FALSE FALSE}, then {@link EXTShaderObject#vkCmdSetSampleLocationsEnableEXT CmdSetSampleLocationsEnableEXT} <b>must</b> have been called and not subsequently <a href="https://registry.khronos.org/vulkan/specs/latest/html/vkspec.html#dynamic-state-lifetime">invalidated</a> in the current command buffer prior to this drawing command</li>
     * <li>If the {@link EXTSampleLocations VK_EXT_sample_locations} extension is enabled, a shader object is bound to any graphics stage or a graphics pipeline is bound which was created with the {@link EXTSampleLocations#VK_DYNAMIC_STATE_SAMPLE_LOCATIONS_EXT DYNAMIC_STATE_SAMPLE_LOCATIONS_EXT} dynamic state enabled, the <a href="https://registry.khronos.org/vulkan/specs/latest/html/vkspec.html#dynamic-state-current-value">current value</a> of {@code rasterizerDiscardEnable} is {@link VK10#VK_FALSE FALSE}, and the <a href="https://registry.khronos.org/vulkan/specs/latest/html/vkspec.html#dynamic-state-current-value">current value</a> of {@code sampleLocationsEnable} is {@link VK10#VK_TRUE TRUE}, then {@link EXTSampleLocations#vkCmdSetSampleLocationsEXT CmdSetSampleLocationsEXT} <b>must</b> have been called and not subsequently <a href="https://registry.khronos.org/vulkan/specs/latest/html/vkspec.html#dynamic-state-lifetime">invalidated</a> in the current command buffer prior to this drawing command</li>
     * <li>If a shader object is bound to any graphics stage or a graphics pipeline is bound which was created with the {@link VK13#VK_DYNAMIC_STATE_CULL_MODE DYNAMIC_STATE_CULL_MODE} dynamic state enabled, and the <a href="https://registry.khronos.org/vulkan/specs/latest/html/vkspec.html#dynamic-state-current-value">current value</a> of {@code rasterizerDiscardEnable} is {@link VK10#VK_FALSE FALSE}, then {@link VK13#vkCmdSetCullMode CmdSetCullMode} <b>must</b> have been called and not subsequently <a href="https://registry.khronos.org/vulkan/specs/latest/html/vkspec.html#dynamic-state-lifetime">invalidated</a> in the current command buffer prior to this drawing command</li>
     * <li>If a shader object is bound to any graphics stage or a graphics pipeline is bound which was created with the {@link VK13#VK_DYNAMIC_STATE_FRONT_FACE DYNAMIC_STATE_FRONT_FACE} dynamic state enabled, and the <a href="https://registry.khronos.org/vulkan/specs/latest/html/vkspec.html#dynamic-state-current-value">current value</a> of {@code rasterizerDiscardEnable} is {@link VK10#VK_FALSE FALSE}, then {@link VK13#vkCmdSetFrontFace CmdSetFrontFace} <b>must</b> have been called and not subsequently <a href="https://registry.khronos.org/vulkan/specs/latest/html/vkspec.html#dynamic-state-lifetime">invalidated</a> in the current command buffer prior to this drawing command</li>
     * <li>If a shader object is bound to any graphics stage or a graphics pipeline is bound which was created with the {@link VK13#VK_DYNAMIC_STATE_DEPTH_TEST_ENABLE DYNAMIC_STATE_DEPTH_TEST_ENABLE} dynamic state enabled, and the <a href="https://registry.khronos.org/vulkan/specs/latest/html/vkspec.html#dynamic-state-current-value">current value</a> of {@code rasterizerDiscardEnable} is {@link VK10#VK_FALSE FALSE}, {@link VK13#vkCmdSetDepthTestEnable CmdSetDepthTestEnable} <b>must</b> have been called and not subsequently <a href="https://registry.khronos.org/vulkan/specs/latest/html/vkspec.html#dynamic-state-lifetime">invalidated</a> in the current command buffer prior to this drawing command</li>
     * <li>If a shader object is bound to any graphics stage or a graphics pipeline is bound which was created with the {@link VK13#VK_DYNAMIC_STATE_DEPTH_WRITE_ENABLE DYNAMIC_STATE_DEPTH_WRITE_ENABLE} dynamic state enabled, and the <a href="https://registry.khronos.org/vulkan/specs/latest/html/vkspec.html#dynamic-state-current-value">current value</a> of {@code rasterizerDiscardEnable} is {@link VK10#VK_FALSE FALSE}, then {@link VK13#vkCmdSetDepthWriteEnable CmdSetDepthWriteEnable} <b>must</b> have been called and not subsequently <a href="https://registry.khronos.org/vulkan/specs/latest/html/vkspec.html#dynamic-state-lifetime">invalidated</a> in the current command buffer prior to this drawing command</li>
     * <li>If a shader object is bound to any graphics stage or a graphics pipeline is bound which was created with the {@link VK13#VK_DYNAMIC_STATE_DEPTH_COMPARE_OP DYNAMIC_STATE_DEPTH_COMPARE_OP} dynamic state enabled, the <a href="https://registry.khronos.org/vulkan/specs/latest/html/vkspec.html#dynamic-state-current-value">current value</a> of {@code rasterizerDiscardEnable} is {@link VK10#VK_FALSE FALSE}, and the <a href="https://registry.khronos.org/vulkan/specs/latest/html/vkspec.html#dynamic-state-current-value">current value</a> of {@code depthTestEnable} is {@link VK10#VK_TRUE TRUE}, then {@link VK13#vkCmdSetDepthCompareOp CmdSetDepthCompareOp} <b>must</b> have been called and not subsequently <a href="https://registry.khronos.org/vulkan/specs/latest/html/vkspec.html#dynamic-state-lifetime">invalidated</a> in the current command buffer prior to this drawing command</li>
     * <li>If the <a href="https://registry.khronos.org/vulkan/specs/latest/html/vkspec.html#features-depthBounds">{@code depthBounds}</a> feature is enabled, a shader object is bound to any graphics stage or a graphics pipeline is bound which was created with the {@link VK13#VK_DYNAMIC_STATE_DEPTH_BOUNDS_TEST_ENABLE DYNAMIC_STATE_DEPTH_BOUNDS_TEST_ENABLE} dynamic state enabled, and the <a href="https://registry.khronos.org/vulkan/specs/latest/html/vkspec.html#dynamic-state-current-value">current value</a> of {@code rasterizerDiscardEnable} is {@link VK10#VK_FALSE FALSE}, then {@link VK13#vkCmdSetDepthBoundsTestEnable CmdSetDepthBoundsTestEnable} <b>must</b> have been called and not subsequently <a href="https://registry.khronos.org/vulkan/specs/latest/html/vkspec.html#dynamic-state-lifetime">invalidated</a> in the current command buffer prior to this drawing command</li>
     * <li>If a shader object is bound to any graphics stage or a graphics pipeline is bound which was created with the {@link VK13#VK_DYNAMIC_STATE_STENCIL_TEST_ENABLE DYNAMIC_STATE_STENCIL_TEST_ENABLE} dynamic state enabled, and the <a href="https://registry.khronos.org/vulkan/specs/latest/html/vkspec.html#dynamic-state-current-value">current value</a> of {@code rasterizerDiscardEnable} is {@link VK10#VK_FALSE FALSE}, then {@link VK13#vkCmdSetStencilTestEnable CmdSetStencilTestEnable} <b>must</b> have been called and not subsequently <a href="https://registry.khronos.org/vulkan/specs/latest/html/vkspec.html#dynamic-state-lifetime">invalidated</a> in the current command buffer prior to this drawing command</li>
     * <li>If a shader object is bound to any graphics stage or a graphics pipeline is bound which was created with the {@link VK13#VK_DYNAMIC_STATE_STENCIL_OP DYNAMIC_STATE_STENCIL_OP} dynamic state enabled, the <a href="https://registry.khronos.org/vulkan/specs/latest/html/vkspec.html#dynamic-state-current-value">current value</a> of {@code rasterizerDiscardEnable} is {@link VK10#VK_FALSE FALSE}, the <a href="https://registry.khronos.org/vulkan/specs/latest/html/vkspec.html#dynamic-state-current-value">current value</a> of {@code stencilTestEnable} is {@link VK10#VK_TRUE TRUE}, then {@link VK13#vkCmdSetStencilOp CmdSetStencilOp} <b>must</b> have been called and not subsequently <a href="https://registry.khronos.org/vulkan/specs/latest/html/vkspec.html#dynamic-state-lifetime">invalidated</a> in the current command buffer prior to this drawing command</li>
     * <li>If a shader object is bound to any graphics stage or a graphics pipeline is bound which was created with the {@link VK13#VK_DYNAMIC_STATE_VIEWPORT_WITH_COUNT DYNAMIC_STATE_VIEWPORT_WITH_COUNT} dynamic state enabled, and the state is not inherited, then {@link VK13#vkCmdSetViewportWithCount CmdSetViewportWithCount} <b>must</b> have been called and not subsequently <a href="https://registry.khronos.org/vulkan/specs/latest/html/vkspec.html#dynamic-state-lifetime">invalidated</a> in the current command buffer prior to this drawing command</li>
     * <li>If a shader object is bound to any graphics stage or a graphics pipeline is bound which was created with the {@link VK13#VK_DYNAMIC_STATE_SCISSOR_WITH_COUNT DYNAMIC_STATE_SCISSOR_WITH_COUNT} dynamic state enabled, and the state is not inherited, then {@link VK13#vkCmdSetScissorWithCount CmdSetScissorWithCount} <b>must</b> have been called and not subsequently <a href="https://registry.khronos.org/vulkan/specs/latest/html/vkspec.html#dynamic-state-lifetime">invalidated</a> in the current command buffer prior to this drawing</li>
     * <li>If a shader object is bound to any graphics stage or a graphics pipeline is bound which was created with both the {@link VK13#VK_DYNAMIC_STATE_SCISSOR_WITH_COUNT DYNAMIC_STATE_SCISSOR_WITH_COUNT} and {@link VK13#VK_DYNAMIC_STATE_VIEWPORT_WITH_COUNT DYNAMIC_STATE_VIEWPORT_WITH_COUNT} dynamic states enabled, and the state is not inherited, then the {@code viewportCount} parameter of {@code vkCmdSetViewportWithCount} <b>must</b> match the {@code scissorCount} parameter of {@code vkCmdSetScissorWithCount}</li>
     * <li>If the bound graphics pipeline state was created with the {@link VK13#VK_DYNAMIC_STATE_VIEWPORT_WITH_COUNT DYNAMIC_STATE_VIEWPORT_WITH_COUNT} dynamic state enabled, but not the {@link NVClipSpaceWScaling#VK_DYNAMIC_STATE_VIEWPORT_W_SCALING_NV DYNAMIC_STATE_VIEWPORT_W_SCALING_NV} dynamic state enabled, then the bound graphics pipeline <b>must</b> have been created with {@link VkPipelineViewportWScalingStateCreateInfoNV}{@code ::viewportCount} greater or equal to the {@code viewportCount} parameter in the last call to {@link VK13#vkCmdSetViewportWithCount CmdSetViewportWithCount}</li>
     * <li>If the {@link NVClipSpaceWScaling VK_NV_clip_space_w_scaling} extension is enabled, and a shader object is bound to any graphics stage or a graphics pipeline is bound which was created with the {@link VK13#VK_DYNAMIC_STATE_VIEWPORT_WITH_COUNT DYNAMIC_STATE_VIEWPORT_WITH_COUNT} and {@link NVClipSpaceWScaling#VK_DYNAMIC_STATE_VIEWPORT_W_SCALING_NV DYNAMIC_STATE_VIEWPORT_W_SCALING_NV} dynamic state enabled, the <a href="https://registry.khronos.org/vulkan/specs/latest/html/vkspec.html#dynamic-state-current-value">current value</a> of {@code viewportWScalingEnable} is {@link VK10#VK_TRUE TRUE}, then {@link NVClipSpaceWScaling#vkCmdSetViewportWScalingNV CmdSetViewportWScalingNV} <b>must</b> have been called and not subsequently <a href="https://registry.khronos.org/vulkan/specs/latest/html/vkspec.html#dynamic-state-lifetime">invalidated</a> in the current command buffer prior to this drawing command</li>
     * <li>If the {@link NVClipSpaceWScaling VK_NV_clip_space_w_scaling} extension is enabled, and a shader object is bound to any graphics stage or a graphics pipeline is bound which was created with the {@link VK13#VK_DYNAMIC_STATE_VIEWPORT_WITH_COUNT DYNAMIC_STATE_VIEWPORT_WITH_COUNT} and {@link NVClipSpaceWScaling#VK_DYNAMIC_STATE_VIEWPORT_W_SCALING_NV DYNAMIC_STATE_VIEWPORT_W_SCALING_NV} dynamic state enabled, the <a href="https://registry.khronos.org/vulkan/specs/latest/html/vkspec.html#dynamic-state-current-value">current value</a> of {@code viewportWScalingEnable} is {@link VK10#VK_TRUE TRUE}, then the {@code viewportCount} parameter in the last call to {@link NVClipSpaceWScaling#vkCmdSetViewportWScalingNV CmdSetViewportWScalingNV} <b>must</b> be greater than or equal to the {@code viewportCount} parameter in the last call to {@link VK13#vkCmdSetViewportWithCount CmdSetViewportWithCount}</li>
     * <li>If the bound graphics pipeline state was created with the {@link VK13#VK_DYNAMIC_STATE_VIEWPORT_WITH_COUNT DYNAMIC_STATE_VIEWPORT_WITH_COUNT} dynamic state enabled, but not the {@link NVShadingRateImage#VK_DYNAMIC_STATE_VIEWPORT_SHADING_RATE_PALETTE_NV DYNAMIC_STATE_VIEWPORT_SHADING_RATE_PALETTE_NV} dynamic state enabled, then the bound graphics pipeline <b>must</b> have been created with {@link VkPipelineViewportShadingRateImageStateCreateInfoNV}{@code ::viewportCount} greater or equal to the {@code viewportCount} parameter in the last call to {@link VK13#vkCmdSetViewportWithCount CmdSetViewportWithCount}</li>
     * <li>If the <a href="https://registry.khronos.org/vulkan/specs/latest/html/vkspec.html#features-shadingRateImage">{@code shadingRateImage}</a> feature is enabled, and a shader object is bound to any graphics stage or a graphics pipeline is bound which was created with the {@link NVShadingRateImage#VK_DYNAMIC_STATE_VIEWPORT_COARSE_SAMPLE_ORDER_NV DYNAMIC_STATE_VIEWPORT_COARSE_SAMPLE_ORDER_NV} and the <a href="https://registry.khronos.org/vulkan/specs/latest/html/vkspec.html#dynamic-state-current-value">current value</a> of {@code rasterizerDiscardEnable} is {@link VK10#VK_FALSE FALSE}, then {@link NVShadingRateImage#vkCmdSetCoarseSampleOrderNV CmdSetCoarseSampleOrderNV} <b>must</b> have been called and not subsequently <a href="https://registry.khronos.org/vulkan/specs/latest/html/vkspec.html#dynamic-state-lifetime">invalidated</a> in the current command buffer prior to this drawing command</li>
     * <li>If the <a href="https://registry.khronos.org/vulkan/specs/latest/html/vkspec.html#features-shadingRateImage">{@code shadingRateImage}</a> feature is enabled, and a shader object is bound to any graphics stage or a graphics pipeline is bound which was created with the {@link VK13#VK_DYNAMIC_STATE_VIEWPORT_WITH_COUNT DYNAMIC_STATE_VIEWPORT_WITH_COUNT} and {@link NVShadingRateImage#VK_DYNAMIC_STATE_VIEWPORT_SHADING_RATE_PALETTE_NV DYNAMIC_STATE_VIEWPORT_SHADING_RATE_PALETTE_NV} dynamic state enabled, the <a href="https://registry.khronos.org/vulkan/specs/latest/html/vkspec.html#dynamic-state-current-value">current value</a> of {@code rasterizerDiscardEnable} is {@link VK10#VK_FALSE FALSE}, and the <a href="https://registry.khronos.org/vulkan/specs/latest/html/vkspec.html#dynamic-state-current-value">current value</a> of {@code shadingRateImageEnable} is {@link VK10#VK_TRUE TRUE}, then {@link NVShadingRateImage#vkCmdSetViewportShadingRatePaletteNV CmdSetViewportShadingRatePaletteNV} <b>must</b> have been called and not subsequently <a href="https://registry.khronos.org/vulkan/specs/latest/html/vkspec.html#dynamic-state-lifetime">invalidated</a> in the current command buffer prior to this drawing command</li>
     * <li>If the <a href="https://registry.khronos.org/vulkan/specs/latest/html/vkspec.html#features-shadingRateImage">{@code shadingRateImage}</a> feature is enabled, and a shader object is bound to any graphics stage or a graphics pipeline is bound which was created with the {@link VK13#VK_DYNAMIC_STATE_VIEWPORT_WITH_COUNT DYNAMIC_STATE_VIEWPORT_WITH_COUNT} and {@link NVShadingRateImage#VK_DYNAMIC_STATE_VIEWPORT_SHADING_RATE_PALETTE_NV DYNAMIC_STATE_VIEWPORT_SHADING_RATE_PALETTE_NV} dynamic state enabled, the <a href="https://registry.khronos.org/vulkan/specs/latest/html/vkspec.html#dynamic-state-current-value">current value</a> of {@code rasterizerDiscardEnable} is {@link VK10#VK_FALSE FALSE}, and the <a href="https://registry.khronos.org/vulkan/specs/latest/html/vkspec.html#dynamic-state-current-value">current value</a> of {@code shadingRateImageEnable} is {@link VK10#VK_TRUE TRUE}, then the {@code viewportCount} parameter in the last call to {@link NVShadingRateImage#vkCmdSetViewportShadingRatePaletteNV CmdSetViewportShadingRatePaletteNV} <b>must</b> be greater than or equal to the {@code viewportCount} parameter in the last call to {@link VK13#vkCmdSetViewportWithCount CmdSetViewportWithCount}</li>
     * <li>If the bound graphics pipeline state was created with the {@link VK13#VK_DYNAMIC_STATE_VIEWPORT_WITH_COUNT DYNAMIC_STATE_VIEWPORT_WITH_COUNT} dynamic state enabled and a {@link VkPipelineViewportSwizzleStateCreateInfoNV} structure chained from {@link VkPipelineViewportStateCreateInfo}, then the bound graphics pipeline <b>must</b> have been created with {@link VkPipelineViewportSwizzleStateCreateInfoNV}{@code ::viewportCount} greater or equal to the {@code viewportCount} parameter in the last call to {@link VK13#vkCmdSetViewportWithCount CmdSetViewportWithCount}</li>
     * <li>If the bound graphics pipeline state was created with the {@link VK13#VK_DYNAMIC_STATE_VIEWPORT_WITH_COUNT DYNAMIC_STATE_VIEWPORT_WITH_COUNT} dynamic state enabled and a {@link VkPipelineViewportExclusiveScissorStateCreateInfoNV} structure chained from {@link VkPipelineViewportStateCreateInfo}, then the bound graphics pipeline <b>must</b> have been created with {@link VkPipelineViewportExclusiveScissorStateCreateInfoNV}{@code ::exclusiveScissorCount} greater or equal to the {@code viewportCount} parameter in the last call to {@link VK13#vkCmdSetViewportWithCount CmdSetViewportWithCount}</li>
     * <li>If the <a href="https://registry.khronos.org/vulkan/specs/latest/html/vkspec.html#features-exclusiveScissor">{@code exclusiveScissor}</a> feature is enabled, and a shader object is bound to any graphics stage or a graphics pipeline is bound which was created with the {@link NVScissorExclusive#VK_DYNAMIC_STATE_EXCLUSIVE_SCISSOR_ENABLE_NV DYNAMIC_STATE_EXCLUSIVE_SCISSOR_ENABLE_NV} dynamic state enabled, then {@link NVScissorExclusive#vkCmdSetExclusiveScissorEnableNV CmdSetExclusiveScissorEnableNV} <b>must</b> have been called and not subsequently <a href="https://registry.khronos.org/vulkan/specs/latest/html/vkspec.html#dynamic-state-lifetime">invalidated</a> in the current command buffer prior to this drawing command</li>
     * <li>If the <a href="https://registry.khronos.org/vulkan/specs/latest/html/vkspec.html#features-exclusiveScissor">{@code exclusiveScissor}</a> feature is enabled, a shader object is bound to any graphics stage or a graphics pipeline is bound which was created with the {@link NVScissorExclusive#VK_DYNAMIC_STATE_EXCLUSIVE_SCISSOR_NV DYNAMIC_STATE_EXCLUSIVE_SCISSOR_NV} dynamic state enabled, and the most recent call to {@link NVScissorExclusive#vkCmdSetExclusiveScissorEnableNV CmdSetExclusiveScissorEnableNV} in the current command buffer set any element of {@code pExclusiveScissorEnables} to {@link VK10#VK_TRUE TRUE}, then {@link NVScissorExclusive#vkCmdSetExclusiveScissorNV CmdSetExclusiveScissorNV} <b>must</b> have been called and not subsequently <a href="https://registry.khronos.org/vulkan/specs/latest/html/vkspec.html#dynamic-state-lifetime">invalidated</a> in the current command buffer prior to this drawing command</li>
     * <li>If a shader object is bound to any graphics stage or a graphics pipeline is bound which was created with the {@link VK13#VK_DYNAMIC_STATE_RASTERIZER_DISCARD_ENABLE DYNAMIC_STATE_RASTERIZER_DISCARD_ENABLE} dynamic state enabled, then {@link VK13#vkCmdSetRasterizerDiscardEnable CmdSetRasterizerDiscardEnable} <b>must</b> have been called and not subsequently <a href="https://registry.khronos.org/vulkan/specs/latest/html/vkspec.html#dynamic-state-lifetime">invalidated</a> in the current command buffer prior to this drawing command</li>
     * <li>If a shader object is bound to any graphics stage or a graphics pipeline is bound which was created with the {@link VK13#VK_DYNAMIC_STATE_DEPTH_BIAS_ENABLE DYNAMIC_STATE_DEPTH_BIAS_ENABLE} dynamic state enabled, and the <a href="https://registry.khronos.org/vulkan/specs/latest/html/vkspec.html#dynamic-state-current-value">current value</a> of {@code rasterizerDiscardEnable} is {@link VK10#VK_FALSE FALSE}, then {@link VK13#vkCmdSetDepthBiasEnable CmdSetDepthBiasEnable} <b>must</b> have been called and not subsequently <a href="https://registry.khronos.org/vulkan/specs/latest/html/vkspec.html#dynamic-state-lifetime">invalidated</a> in the current command buffer prior to this drawing command</li>
     * <li>If a shader object is bound to the {@link VK10#VK_SHADER_STAGE_FRAGMENT_BIT SHADER_STAGE_FRAGMENT_BIT} or a graphics pipeline is bound which was created with the {@link EXTExtendedDynamicState2#VK_DYNAMIC_STATE_LOGIC_OP_EXT DYNAMIC_STATE_LOGIC_OP_EXT} dynamic state enabled, the <a href="https://registry.khronos.org/vulkan/specs/latest/html/vkspec.html#dynamic-state-current-value">current value</a> of {@code rasterizerDiscardEnable} is {@link VK10#VK_FALSE FALSE}, and the <a href="https://registry.khronos.org/vulkan/specs/latest/html/vkspec.html#dynamic-state-current-value">current value</a> of {@code logicOpEnable} is {@link VK10#VK_TRUE TRUE}, then {@link EXTShaderObject#vkCmdSetLogicOpEXT CmdSetLogicOpEXT} <b>must</b> have been called and not subsequently <a href="https://registry.khronos.org/vulkan/specs/latest/html/vkspec.html#dynamic-state-lifetime">invalidated</a> in the current command buffer prior to this drawing command</li>
     * <li>If the <a href="https://registry.khronos.org/vulkan/specs/latest/html/vkspec.html#limits-primitiveFragmentShadingRateWithMultipleViewports">{@code primitiveFragmentShadingRateWithMultipleViewports}</a> limit is not supported, the bound graphics pipeline was created with the {@link VK13#VK_DYNAMIC_STATE_VIEWPORT_WITH_COUNT DYNAMIC_STATE_VIEWPORT_WITH_COUNT} dynamic state enabled, and any of the shader stages of the bound graphics pipeline write to the {@code PrimitiveShadingRateKHR} built-in, then {@link VK13#vkCmdSetViewportWithCount CmdSetViewportWithCount} <b>must</b> have been called in the current command buffer prior to this drawing command, and the {@code viewportCount} parameter of {@code vkCmdSetViewportWithCount} <b>must</b> be 1</li>
     * <li>If the <a href="https://registry.khronos.org/vulkan/specs/latest/html/vkspec.html#limits-primitiveFragmentShadingRateWithMultipleViewports">{@code primitiveFragmentShadingRateWithMultipleViewports}</a> limit is not supported, and any shader object bound to a graphics stage writes to the {@code PrimitiveShadingRateKHR} built-in, then {@link VK13#vkCmdSetViewportWithCount CmdSetViewportWithCount} <b>must</b> have been called in the current command buffer prior to this drawing command, and the {@code viewportCount} parameter of {@code vkCmdSetViewportWithCount} <b>must</b> be 1</li>
     * <li>If rasterization is not disabled in the bound graphics pipeline, then for each color attachment in the subpass, if the corresponding image view’s <a href="https://registry.khronos.org/vulkan/specs/latest/html/vkspec.html#resources-image-view-format-features">format features</a> do not contain {@link VK10#VK_FORMAT_FEATURE_COLOR_ATTACHMENT_BLEND_BIT FORMAT_FEATURE_COLOR_ATTACHMENT_BLEND_BIT}, then the {@code blendEnable} member of the corresponding element of the {@code pAttachments} member of {@code pColorBlendState} <b>must</b> be {@link VK10#VK_FALSE FALSE}</li>
     * <li>If a shader object is bound to the {@link VK10#VK_SHADER_STAGE_FRAGMENT_BIT SHADER_STAGE_FRAGMENT_BIT} stage, and the most recent call to {@link VK13#vkCmdSetRasterizerDiscardEnable CmdSetRasterizerDiscardEnable} in the current command buffer set {@code rasterizerDiscardEnable} to {@link VK10#VK_FALSE FALSE}, then for each color attachment in the render pass, if the corresponding image view’s <a href="https://registry.khronos.org/vulkan/specs/latest/html/vkspec.html#resources-image-view-format-features">format features</a> do not contain {@link VK10#VK_FORMAT_FEATURE_COLOR_ATTACHMENT_BLEND_BIT FORMAT_FEATURE_COLOR_ATTACHMENT_BLEND_BIT}, then the corresponding member of {@code pColorBlendEnables} in the most recent call to {@code vkCmdSetColorBlendEnableEXT} in the current command buffer that affected that attachment index <b>must</b> have been {@link VK10#VK_FALSE FALSE}</li>
     * <li>If rasterization is not disabled in the bound graphics pipeline, and none of the following is enabled:
     * 
     * <ul>
     * <li>the {@link AMDMixedAttachmentSamples VK_AMD_mixed_attachment_samples} extension</li>
     * <li>the {@link NVFramebufferMixedSamples VK_NV_framebuffer_mixed_samples} extension</li>
     * <li>the <a href="https://registry.khronos.org/vulkan/specs/latest/html/vkspec.html#features-multisampledRenderToSingleSampled">{@code multisampledRenderToSingleSampled}</a> feature</li>
     * </ul>
     * 
     * <p>then {@code rasterizationSamples} for the bound graphics pipeline <b>must</b> be the same as the current subpass color and/or depth/stencil attachments</p>
     * </li>
     * <li>If a shader object is bound to any graphics stage, and the most recent call to {@link VK13#vkCmdSetRasterizerDiscardEnable CmdSetRasterizerDiscardEnable} in the current command buffer set {@code rasterizerDiscardEnable} to {@link VK10#VK_FALSE FALSE}, and none of the following is enabled:
     * 
     * <ul>
     * <li>the {@link AMDMixedAttachmentSamples VK_AMD_mixed_attachment_samples} extension</li>
     * <li>the {@link NVFramebufferMixedSamples VK_NV_framebuffer_mixed_samples} extension</li>
     * <li>the <a href="https://registry.khronos.org/vulkan/specs/latest/html/vkspec.html#features-multisampledRenderToSingleSampled">{@code multisampledRenderToSingleSampled}</a> feature</li>
     * </ul>
     * 
     * <p>then the most recent call to {@link EXTShaderObject#vkCmdSetRasterizationSamplesEXT CmdSetRasterizationSamplesEXT} in the current command buffer <b>must</b> have set {@code rasterizationSamples} to be the same as the number of samples for the current render pass color and/or depth/stencil attachments</p>
     * </li>
     * <li>If a shader object is bound to any graphics stage, the current render pass instance <b>must</b> have been begun with {@link VK13#vkCmdBeginRendering CmdBeginRendering}</li>
     * <li>If the current render pass instance was begun with {@link VK13#vkCmdBeginRendering CmdBeginRendering}, the {@code imageView} member of {@code pDepthAttachment} is not {@link VK10#VK_NULL_HANDLE NULL_HANDLE}, and the {@code layout} member of {@code pDepthAttachment} is {@link VK10#VK_IMAGE_LAYOUT_DEPTH_STENCIL_READ_ONLY_OPTIMAL IMAGE_LAYOUT_DEPTH_STENCIL_READ_ONLY_OPTIMAL}, this command <b>must</b> not write any values to the depth attachment</li>
     * <li>If the current render pass instance was begun with {@link VK13#vkCmdBeginRendering CmdBeginRendering}, the {@code imageView} member of {@code pStencilAttachment} is not {@link VK10#VK_NULL_HANDLE NULL_HANDLE}, and the {@code layout} member of {@code pStencilAttachment} is {@link VK10#VK_IMAGE_LAYOUT_DEPTH_STENCIL_READ_ONLY_OPTIMAL IMAGE_LAYOUT_DEPTH_STENCIL_READ_ONLY_OPTIMAL}, this command <b>must</b> not write any values to the stencil attachment</li>
     * <li>If the current render pass instance was begun with {@link VK13#vkCmdBeginRendering CmdBeginRendering}, the {@code imageView} member of {@code pDepthAttachment} is not {@link VK10#VK_NULL_HANDLE NULL_HANDLE}, and the {@code layout} member of {@code pDepthAttachment} is {@link VK11#VK_IMAGE_LAYOUT_DEPTH_READ_ONLY_STENCIL_ATTACHMENT_OPTIMAL IMAGE_LAYOUT_DEPTH_READ_ONLY_STENCIL_ATTACHMENT_OPTIMAL}, this command <b>must</b> not write any values to the depth attachment</li>
     * <li>If the current render pass instance was begun with {@link VK13#vkCmdBeginRendering CmdBeginRendering}, the {@code imageView} member of {@code pStencilAttachment} is not {@link VK10#VK_NULL_HANDLE NULL_HANDLE}, and the {@code layout} member of {@code pStencilAttachment} is {@link VK11#VK_IMAGE_LAYOUT_DEPTH_ATTACHMENT_STENCIL_READ_ONLY_OPTIMAL IMAGE_LAYOUT_DEPTH_ATTACHMENT_STENCIL_READ_ONLY_OPTIMAL}, this command <b>must</b> not write any values to the stencil attachment</li>
     * <li>If the current render pass instance was begun with {@link VK13#vkCmdBeginRendering CmdBeginRendering}, the {@code imageView} member of {@code pDepthAttachment} is not {@link VK10#VK_NULL_HANDLE NULL_HANDLE}, and the {@code layout} member of {@code pDepthAttachment} is {@link VK12#VK_IMAGE_LAYOUT_DEPTH_READ_ONLY_OPTIMAL IMAGE_LAYOUT_DEPTH_READ_ONLY_OPTIMAL}, this command <b>must</b> not write any values to the depth attachment</li>
     * <li>If the current render pass instance was begun with {@link VK13#vkCmdBeginRendering CmdBeginRendering}, the {@code imageView} member of {@code pStencilAttachment} is not {@link VK10#VK_NULL_HANDLE NULL_HANDLE}, and the {@code layout} member of {@code pStencilAttachment} is {@link VK12#VK_IMAGE_LAYOUT_STENCIL_READ_ONLY_OPTIMAL IMAGE_LAYOUT_STENCIL_READ_ONLY_OPTIMAL}, this command <b>must</b> not write any values to the stencil attachment</li>
     * <li>If the current render pass instance was begun with {@link VK13#vkCmdBeginRendering CmdBeginRendering}, the bound graphics pipeline <b>must</b> have been created with a {@link VkPipelineRenderingCreateInfo}{@code ::viewMask} equal to {@link VkRenderingInfo}{@code ::viewMask}</li>
     * <li>If the <a href="https://registry.khronos.org/vulkan/specs/latest/html/vkspec.html#features-dynamicRenderingUnusedAttachments">{@code dynamicRenderingUnusedAttachments}</a> feature is not enabled and the current render pass instance was begun with {@link VK13#vkCmdBeginRendering CmdBeginRendering}, the bound graphics pipeline <b>must</b> have been created with a {@link VkPipelineRenderingCreateInfo}{@code ::colorAttachmentCount} equal to {@link VkRenderingInfo}{@code ::colorAttachmentCount}</li>
     * <li>If the <a href="https://registry.khronos.org/vulkan/specs/latest/html/vkspec.html#features-dynamicRenderingUnusedAttachments">{@code dynamicRenderingUnusedAttachments}</a> feature is not enabled, and the current render pass instance was begun with {@link VK13#vkCmdBeginRendering CmdBeginRendering} and {@link VkRenderingInfo}{@code ::colorAttachmentCount} greater than 0, then each element of the {@link VkRenderingInfo}{@code ::pColorAttachments} array with an {@code imageView} not equal to {@link VK10#VK_NULL_HANDLE NULL_HANDLE} <b>must</b> have been created with a {@code VkFormat} equal to the corresponding element of {@link VkPipelineRenderingCreateInfo}{@code ::pColorAttachmentFormats} used to create the bound graphics pipeline</li>
     * <li>If the <a href="https://registry.khronos.org/vulkan/specs/latest/html/vkspec.html#features-dynamicRenderingUnusedAttachments">{@code dynamicRenderingUnusedAttachments}</a> feature is not enabled, and the current render pass instance was begun with {@link VK13#vkCmdBeginRendering CmdBeginRendering} and {@link VkRenderingInfo}{@code ::colorAttachmentCount} greater than 0, then each element of the {@link VkRenderingInfo}{@code ::pColorAttachments} array with an {@code imageView} equal to {@link VK10#VK_NULL_HANDLE NULL_HANDLE} <b>must</b> have the corresponding element of {@link VkPipelineRenderingCreateInfo}{@code ::pColorAttachmentFormats} used to create the bound pipeline equal to {@link VK10#VK_FORMAT_UNDEFINED FORMAT_UNDEFINED}</li>
     * <li>If the <a href="https://registry.khronos.org/vulkan/specs/latest/html/vkspec.html#features-dynamicRenderingUnusedAttachments">{@code dynamicRenderingUnusedAttachments}</a> feature is enabled, and the current render pass instance was begun with {@link VK13#vkCmdBeginRendering CmdBeginRendering} and {@link VkRenderingInfo}{@code ::colorAttachmentCount} greater than 0, then each element of the {@link VkRenderingInfo}{@code ::pColorAttachments} array with an {@code imageView} not equal to {@link VK10#VK_NULL_HANDLE NULL_HANDLE} <b>must</b> have been created with a {@code VkFormat} equal to the corresponding element of {@link VkPipelineRenderingCreateInfo}{@code ::pColorAttachmentFormats} used to create the bound graphics pipeline, or the corresponding element of {@link VkPipelineRenderingCreateInfo}{@code ::pColorAttachmentFormats}, if it exists, <b>must</b> be {@link VK10#VK_FORMAT_UNDEFINED FORMAT_UNDEFINED}</li>
     * <li>If the current render pass instance was begun with {@link VK13#vkCmdBeginRendering CmdBeginRendering}, with a {@link VkRenderingInfo}{@code ::colorAttachmentCount} equal to 1, there is no shader object bound to any graphics stage, and a color attachment with a resolve mode of {@link ANDROIDExternalFormatResolve#VK_RESOLVE_MODE_EXTERNAL_FORMAT_DOWNSAMPLE_ANDROID RESOLVE_MODE_EXTERNAL_FORMAT_DOWNSAMPLE_ANDROID}, each element of the {@link VkRenderingInfo}{@code ::pColorAttachments} array with a {@code resolveImageView} not equal to {@link VK10#VK_NULL_HANDLE NULL_HANDLE} <b>must</b> have been created with an image created with a {@link VkExternalFormatANDROID}{@code ::externalFormat} value equal to the {@link VkExternalFormatANDROID}{@code ::externalFormat} value used to create the bound graphics pipeline</li>
     * <li>If there is no shader object bound to any graphics stage, the current render pass instance was begun with {@link VK13#vkCmdBeginRendering CmdBeginRendering} and a {@link VkRenderingInfo}{@code ::colorAttachmentCount} equal to 1, and a color attachment with a resolve mode of {@link ANDROIDExternalFormatResolve#VK_RESOLVE_MODE_EXTERNAL_FORMAT_DOWNSAMPLE_ANDROID RESOLVE_MODE_EXTERNAL_FORMAT_DOWNSAMPLE_ANDROID}, each element of the {@link VkRenderingInfo}{@code ::pColorAttachments} array with a {@code imageView} not equal to {@link VK10#VK_NULL_HANDLE NULL_HANDLE} <b>must</b> have been created with an image created with a {@link VkExternalFormatANDROID}{@code ::externalFormat} value equal to the {@link VkExternalFormatANDROID}{@code ::externalFormat} value used to create the bound graphics pipeline</li>
     * <li>If the current render pass instance was begun with {@link VK13#vkCmdBeginRendering CmdBeginRendering}, there is no shader object bound to any graphics stage, and the bound graphics pipeline was created with a non-zero {@link VkExternalFormatANDROID}{@code ::externalFormat} value and with the {@link EXTExtendedDynamicState3#VK_DYNAMIC_STATE_COLOR_BLEND_ENABLE_EXT DYNAMIC_STATE_COLOR_BLEND_ENABLE_EXT} dynamic state enabled, then {@link EXTShaderObject#vkCmdSetColorBlendEnableEXT CmdSetColorBlendEnableEXT} <b>must</b> have set the blend enable to {@link VK10#VK_FALSE FALSE} prior to this drawing command</li>
     * <li>If the current render pass instance was begun with {@link VK13#vkCmdBeginRendering CmdBeginRendering}, there is no shader object bound to any graphics stage, and the bound graphics pipeline was created with a non-zero {@link VkExternalFormatANDROID}{@code ::externalFormat} value and with the {@link EXTExtendedDynamicState3#VK_DYNAMIC_STATE_RASTERIZATION_SAMPLES_EXT DYNAMIC_STATE_RASTERIZATION_SAMPLES_EXT} dynamic state enabled, then {@link EXTShaderObject#vkCmdSetRasterizationSamplesEXT CmdSetRasterizationSamplesEXT} <b>must</b> have set {@code rasterizationSamples} to {@link VK10#VK_SAMPLE_COUNT_1_BIT SAMPLE_COUNT_1_BIT} prior to this drawing command</li>
     * <li>If there is a shader object bound to any graphics stage, and the current render pass includes a color attachment that uses the {@link ANDROIDExternalFormatResolve#VK_RESOLVE_MODE_EXTERNAL_FORMAT_DOWNSAMPLE_ANDROID RESOLVE_MODE_EXTERNAL_FORMAT_DOWNSAMPLE_ANDROID} resolve mode, then {@link EXTShaderObject#vkCmdSetColorBlendEnableEXT CmdSetColorBlendEnableEXT} <b>must</b> have set blend enable to {@link VK10#VK_FALSE FALSE} prior to this drawing command</li>
     * <li>If there is a shader object bound to any graphics stage, and the current render pass includes a color attachment that uses the {@link ANDROIDExternalFormatResolve#VK_RESOLVE_MODE_EXTERNAL_FORMAT_DOWNSAMPLE_ANDROID RESOLVE_MODE_EXTERNAL_FORMAT_DOWNSAMPLE_ANDROID} resolve mode, then {@link EXTShaderObject#vkCmdSetRasterizationSamplesEXT CmdSetRasterizationSamplesEXT} <b>must</b> have set {@code rasterizationSamples} to {@link VK10#VK_SAMPLE_COUNT_1_BIT SAMPLE_COUNT_1_BIT} prior to this drawing command</li>
     * <li>If the current render pass instance was begun with {@link VK13#vkCmdBeginRendering CmdBeginRendering}, there is no shader object bound to any graphics stage, and the bound graphics pipeline was created with a non-zero {@link VkExternalFormatANDROID}{@code ::externalFormat} value and with the {@link KHRFragmentShadingRate#VK_DYNAMIC_STATE_FRAGMENT_SHADING_RATE_KHR DYNAMIC_STATE_FRAGMENT_SHADING_RATE_KHR} dynamic state enabled, then {@link KHRFragmentShadingRate#vkCmdSetFragmentShadingRateKHR CmdSetFragmentShadingRateKHR} <b>must</b> have set {@code pFragmentSize→width} to 1 prior to this drawing command</li>
     * <li>If the current render pass instance was begun with {@link VK13#vkCmdBeginRendering CmdBeginRendering}, there is no shader object bound to any graphics stage, and the bound graphics pipeline was created with a non-zero {@link VkExternalFormatANDROID}{@code ::externalFormat} value and with the {@link KHRFragmentShadingRate#VK_DYNAMIC_STATE_FRAGMENT_SHADING_RATE_KHR DYNAMIC_STATE_FRAGMENT_SHADING_RATE_KHR} dynamic state enabled, then {@link KHRFragmentShadingRate#vkCmdSetFragmentShadingRateKHR CmdSetFragmentShadingRateKHR} <b>must</b> have set {@code pFragmentSize→height} to 1 prior to this drawing command</li>
     * <li>If there is a shader object bound to any graphics stage, and the current render pass includes a color attachment that uses the {@link ANDROIDExternalFormatResolve#VK_RESOLVE_MODE_EXTERNAL_FORMAT_DOWNSAMPLE_ANDROID RESOLVE_MODE_EXTERNAL_FORMAT_DOWNSAMPLE_ANDROID} resolve mode, then {@link KHRFragmentShadingRate#vkCmdSetFragmentShadingRateKHR CmdSetFragmentShadingRateKHR} <b>must</b> have set {@code pFragmentSize→width} to 1 prior to this drawing command</li>
     * <li>If there is a shader object bound to any graphics stage, and the current render pass includes a color attachment that uses the {@link ANDROIDExternalFormatResolve#VK_RESOLVE_MODE_EXTERNAL_FORMAT_DOWNSAMPLE_ANDROID RESOLVE_MODE_EXTERNAL_FORMAT_DOWNSAMPLE_ANDROID} resolve mode, then {@link KHRFragmentShadingRate#vkCmdSetFragmentShadingRateKHR CmdSetFragmentShadingRateKHR} <b>must</b> have set {@code pFragmentSize→height} to 1 prior to this drawing command</li>
     * <li>If the bound graphics pipeline state was created with the {@link EXTColorWriteEnable#VK_DYNAMIC_STATE_COLOR_WRITE_ENABLE_EXT DYNAMIC_STATE_COLOR_WRITE_ENABLE_EXT} dynamic state enabled then {@link EXTColorWriteEnable#vkCmdSetColorWriteEnableEXT CmdSetColorWriteEnableEXT} <b>must</b> have been called and not subsequently <a href="https://registry.khronos.org/vulkan/specs/latest/html/vkspec.html#dynamic-state-lifetime">invalidated</a> in the current command buffer prior to this drawing command</li>
     * <li>If the <a href="https://registry.khronos.org/vulkan/specs/latest/html/vkspec.html#features-colorWriteEnable">{@code colorWriteEnable}</a> feature is enabled, and a shader object is bound to the {@link VK10#VK_SHADER_STAGE_FRAGMENT_BIT SHADER_STAGE_FRAGMENT_BIT} stage, and the most recent call to {@link VK13#vkCmdSetRasterizerDiscardEnable CmdSetRasterizerDiscardEnable} in the current command buffer set {@code rasterizerDiscardEnable} to {@link VK10#VK_FALSE FALSE}, then {@link EXTColorWriteEnable#vkCmdSetColorWriteEnableEXT CmdSetColorWriteEnableEXT} <b>must</b> have been called and not subsequently <a href="https://registry.khronos.org/vulkan/specs/latest/html/vkspec.html#dynamic-state-lifetime">invalidated</a> in the current command buffer prior to this drawing command</li>
     * <li>If the bound graphics pipeline state was created with the {@link EXTColorWriteEnable#VK_DYNAMIC_STATE_COLOR_WRITE_ENABLE_EXT DYNAMIC_STATE_COLOR_WRITE_ENABLE_EXT} dynamic state enabled then the {@code attachmentCount} parameter of {@code vkCmdSetColorWriteEnableEXT} <b>must</b> be greater than or equal to the {@link VkPipelineColorBlendStateCreateInfo}{@code ::attachmentCount} of the bound graphics pipeline</li>
     * <li>If the <a href="https://registry.khronos.org/vulkan/specs/latest/html/vkspec.html#features-colorWriteEnable">{@code colorWriteEnable}</a> feature is enabled, and a shader object is bound to the {@link VK10#VK_SHADER_STAGE_FRAGMENT_BIT SHADER_STAGE_FRAGMENT_BIT} stage, and the most recent call to {@link VK13#vkCmdSetRasterizerDiscardEnable CmdSetRasterizerDiscardEnable} in the current command buffer set {@code rasterizerDiscardEnable} to {@link VK10#VK_FALSE FALSE}, then the {@code attachmentCount} parameter of most recent call to {@code vkCmdSetColorWriteEnableEXT} in the current command buffer <b>must</b> be greater than or equal to the number of color attachments in the current render pass instance</li>
     * <li>If the bound graphics pipeline state was created with the {@link EXTDiscardRectangles#VK_DYNAMIC_STATE_DISCARD_RECTANGLE_EXT DYNAMIC_STATE_DISCARD_RECTANGLE_EXT} dynamic state enabled then {@link EXTDiscardRectangles#vkCmdSetDiscardRectangleEXT CmdSetDiscardRectangleEXT} <b>must</b> have been called and not subsequently <a href="https://registry.khronos.org/vulkan/specs/latest/html/vkspec.html#dynamic-state-lifetime">invalidated</a> in the current command buffer prior to this drawing command for each discard rectangle in {@link VkPipelineDiscardRectangleStateCreateInfoEXT}{@code ::discardRectangleCount}</li>
     * <li>If the {@link EXTDiscardRectangles VK_EXT_discard_rectangles} extension is enabled, a shader object is bound to any graphics stage or a graphics pipeline is bound which was created with the {@link EXTDiscardRectangles#VK_DYNAMIC_STATE_DISCARD_RECTANGLE_ENABLE_EXT DYNAMIC_STATE_DISCARD_RECTANGLE_ENABLE_EXT} dynamic state enabled, the <a href="https://registry.khronos.org/vulkan/specs/latest/html/vkspec.html#dynamic-state-current-value">current value</a> of {@code rasterizerDiscardEnable} is {@link VK10#VK_FALSE FALSE}, then {@link EXTDiscardRectangles#vkCmdSetDiscardRectangleEnableEXT CmdSetDiscardRectangleEnableEXT} <b>must</b> have been called and not subsequently <a href="https://registry.khronos.org/vulkan/specs/latest/html/vkspec.html#dynamic-state-lifetime">invalidated</a> in the current command buffer prior to this drawing command</li>
     * <li>If the {@link EXTDiscardRectangles VK_EXT_discard_rectangles} extension is enabled, a shader object is bound to any graphics stage or a graphics pipeline is bound which was created with the {@link EXTDiscardRectangles#VK_DYNAMIC_STATE_DISCARD_RECTANGLE_MODE_EXT DYNAMIC_STATE_DISCARD_RECTANGLE_MODE_EXT} dynamic state enabled, the <a href="https://registry.khronos.org/vulkan/specs/latest/html/vkspec.html#dynamic-state-current-value">current value</a> of {@code rasterizerDiscardEnable} is {@link VK10#VK_FALSE FALSE}, and the <a href="https://registry.khronos.org/vulkan/specs/latest/html/vkspec.html#dynamic-state-current-value">current value</a> of {@code discardRectangleEnable} is {@link VK10#VK_TRUE TRUE}, then {@link EXTDiscardRectangles#vkCmdSetDiscardRectangleModeEXT CmdSetDiscardRectangleModeEXT} <b>must</b> have been called and not subsequently <a href="https://registry.khronos.org/vulkan/specs/latest/html/vkspec.html#dynamic-state-lifetime">invalidated</a> in the current command buffer prior to this drawing command</li>
     * <li>If the {@link EXTDiscardRectangles VK_EXT_discard_rectangles} extension is enabled, and a shader object is bound to any graphics stage, and the most recent call to {@link VK13#vkCmdSetRasterizerDiscardEnable CmdSetRasterizerDiscardEnable} in the current command buffer set {@code rasterizerDiscardEnable} to {@link VK10#VK_FALSE FALSE}, and the most recent call to {@link EXTDiscardRectangles#vkCmdSetDiscardRectangleEnableEXT CmdSetDiscardRectangleEnableEXT} in the current command buffer set {@code discardRectangleEnable} to {@link VK10#VK_TRUE TRUE}, then {@link EXTDiscardRectangles#vkCmdSetDiscardRectangleEXT CmdSetDiscardRectangleEXT} <b>must</b> have been called and not subsequently <a href="https://registry.khronos.org/vulkan/specs/latest/html/vkspec.html#dynamic-state-lifetime">invalidated</a> in the current command buffer prior to this drawing command</li>
     * <li>If the current render pass instance was begun with {@link VK13#vkCmdBeginRendering CmdBeginRendering}, the <a href="https://registry.khronos.org/vulkan/specs/latest/html/vkspec.html#features-dynamicRenderingUnusedAttachments">{@code dynamicRenderingUnusedAttachments}</a> feature is not enabled, and {@link VkRenderingInfo}{@code ::pDepthAttachment→imageView} was {@link VK10#VK_NULL_HANDLE NULL_HANDLE}, the value of {@link VkPipelineRenderingCreateInfo}{@code ::depthAttachmentFormat} used to create the bound graphics pipeline <b>must</b> be equal to {@link VK10#VK_FORMAT_UNDEFINED FORMAT_UNDEFINED}</li>
     * <li>If current render pass instance was begun with {@link VK13#vkCmdBeginRendering CmdBeginRendering}, the <a href="https://registry.khronos.org/vulkan/specs/latest/html/vkspec.html#features-dynamicRenderingUnusedAttachments">{@code dynamicRenderingUnusedAttachments}</a> feature is not enabled, and {@link VkRenderingInfo}{@code ::pDepthAttachment→imageView} was not {@link VK10#VK_NULL_HANDLE NULL_HANDLE}, the value of {@link VkPipelineRenderingCreateInfo}{@code ::depthAttachmentFormat} used to create the bound graphics pipeline <b>must</b> be equal to the {@code VkFormat} used to create {@link VkRenderingInfo}{@code ::pDepthAttachment→imageView}</li>
     * <li>If the current render pass instance was begun with {@link VK13#vkCmdBeginRendering CmdBeginRendering}, the <a href="https://registry.khronos.org/vulkan/specs/latest/html/vkspec.html#features-dynamicRenderingUnusedAttachments">{@code dynamicRenderingUnusedAttachments}</a> feature is enabled, {@link VkRenderingInfo}{@code ::pDepthAttachment→imageView} was not {@link VK10#VK_NULL_HANDLE NULL_HANDLE}, and the value of {@link VkPipelineRenderingCreateInfo}{@code ::depthAttachmentFormat} used to create the bound graphics pipeline was not equal to the {@code VkFormat} used to create {@link VkRenderingInfo}{@code ::pDepthAttachment→imageView}, the value of the format <b>must</b> be {@link VK10#VK_FORMAT_UNDEFINED FORMAT_UNDEFINED}</li>
     * <li>If the current render pass instance was begun with {@link VK13#vkCmdBeginRendering CmdBeginRendering}, the <a href="https://registry.khronos.org/vulkan/specs/latest/html/vkspec.html#features-dynamicRenderingUnusedAttachments">{@code dynamicRenderingUnusedAttachments}</a> feature is not enabled, and {@link VkRenderingInfo}{@code ::pStencilAttachment→imageView} was {@link VK10#VK_NULL_HANDLE NULL_HANDLE}, the value of {@link VkPipelineRenderingCreateInfo}{@code ::stencilAttachmentFormat} used to create the bound graphics pipeline <b>must</b> be equal to {@link VK10#VK_FORMAT_UNDEFINED FORMAT_UNDEFINED}</li>
     * <li>If current render pass instance was begun with {@link VK13#vkCmdBeginRendering CmdBeginRendering}, the <a href="https://registry.khronos.org/vulkan/specs/latest/html/vkspec.html#features-dynamicRenderingUnusedAttachments">{@code dynamicRenderingUnusedAttachments}</a> feature is not enabled, and {@link VkRenderingInfo}{@code ::pStencilAttachment→imageView} was not {@link VK10#VK_NULL_HANDLE NULL_HANDLE}, the value of {@link VkPipelineRenderingCreateInfo}{@code ::stencilAttachmentFormat} used to create the bound graphics pipeline <b>must</b> be equal to the {@code VkFormat} used to create {@link VkRenderingInfo}{@code ::pStencilAttachment→imageView}</li>
     * <li>If the current render pass instance was begun with {@link VK13#vkCmdBeginRendering CmdBeginRendering}, the <a href="https://registry.khronos.org/vulkan/specs/latest/html/vkspec.html#features-dynamicRenderingUnusedAttachments">{@code dynamicRenderingUnusedAttachments}</a> feature is enabled, {@link VkRenderingInfo}{@code ::pStencilAttachment→imageView} was not {@link VK10#VK_NULL_HANDLE NULL_HANDLE}, and the value of {@link VkPipelineRenderingCreateInfo}{@code ::stencilAttachmentFormat} used to create the bound graphics pipeline was not equal to the {@code VkFormat} used to create {@link VkRenderingInfo}{@code ::pStencilAttachment→imageView}, the value of the format <b>must</b> be {@link VK10#VK_FORMAT_UNDEFINED FORMAT_UNDEFINED}</li>
     * <li>If the current render pass instance was begun with {@link VK13#vkCmdBeginRendering CmdBeginRendering} and {@link VkRenderingFragmentShadingRateAttachmentInfoKHR}{@code ::imageView} was not {@link VK10#VK_NULL_HANDLE NULL_HANDLE}, the bound graphics pipeline <b>must</b> have been created with {@link KHRFragmentShadingRate#VK_PIPELINE_CREATE_RENDERING_FRAGMENT_SHADING_RATE_ATTACHMENT_BIT_KHR PIPELINE_CREATE_RENDERING_FRAGMENT_SHADING_RATE_ATTACHMENT_BIT_KHR}</li>
     * <li>If the current render pass instance was begun with {@link VK13#vkCmdBeginRendering CmdBeginRendering} and {@link VkRenderingFragmentDensityMapAttachmentInfoEXT}{@code ::imageView} was not {@link VK10#VK_NULL_HANDLE NULL_HANDLE}, the bound graphics pipeline <b>must</b> have been created with {@link EXTFragmentDensityMap#VK_PIPELINE_CREATE_RENDERING_FRAGMENT_DENSITY_MAP_ATTACHMENT_BIT_EXT PIPELINE_CREATE_RENDERING_FRAGMENT_DENSITY_MAP_ATTACHMENT_BIT_EXT}</li>
     * <li>If the bound pipeline was created with a {@link VkAttachmentSampleCountInfoAMD} or {@link VkAttachmentSampleCountInfoNV} structure, and the current render pass instance was begun with {@link VK13#vkCmdBeginRendering CmdBeginRendering} with a {@link VkRenderingInfo}{@code ::colorAttachmentCount} parameter greater than 0, then each element of the {@link VkRenderingInfo}{@code ::pColorAttachments} array with a {@code imageView} not equal to {@link VK10#VK_NULL_HANDLE NULL_HANDLE} <b>must</b> have been created with a sample count equal to the corresponding element of the {@code pColorAttachmentSamples} member of {@link VkAttachmentSampleCountInfoAMD} or {@link VkAttachmentSampleCountInfoNV} used to create the bound graphics pipeline</li>
     * <li>If the current render pass instance was begun with {@link VK13#vkCmdBeginRendering CmdBeginRendering}, the bound pipeline was created with a {@link VkAttachmentSampleCountInfoAMD} or {@link VkAttachmentSampleCountInfoNV} structure, and {@link VkRenderingInfo}{@code ::pDepthAttachment→imageView} was not {@link VK10#VK_NULL_HANDLE NULL_HANDLE}, the value of the {@code depthStencilAttachmentSamples} member of {@link VkAttachmentSampleCountInfoAMD} or {@link VkAttachmentSampleCountInfoNV} used to create the bound graphics pipeline <b>must</b> be equal to the sample count used to create {@link VkRenderingInfo}{@code ::pDepthAttachment→imageView}</li>
     * <li>If the current render pass instance was begun with {@link VK13#vkCmdBeginRendering CmdBeginRendering}, the bound pipeline was created with a {@link VkAttachmentSampleCountInfoAMD} or {@link VkAttachmentSampleCountInfoNV} structure, and {@link VkRenderingInfo}{@code ::pStencilAttachment→imageView} was not {@link VK10#VK_NULL_HANDLE NULL_HANDLE}, the value of the {@code depthStencilAttachmentSamples} member of {@link VkAttachmentSampleCountInfoAMD} or {@link VkAttachmentSampleCountInfoNV} used to create the bound graphics pipeline <b>must</b> be equal to the sample count used to create {@link VkRenderingInfo}{@code ::pStencilAttachment→imageView}</li>
     * <li>If the bound pipeline was created without a {@link VkAttachmentSampleCountInfoAMD} or {@link VkAttachmentSampleCountInfoNV} structure, and the <a href="https://registry.khronos.org/vulkan/specs/latest/html/vkspec.html#features-multisampledRenderToSingleSampled">{@code multisampledRenderToSingleSampled}</a> feature is not enabled, and the current render pass instance was begun with {@link VK13#vkCmdBeginRendering CmdBeginRendering} with a {@link VkRenderingInfo}{@code ::colorAttachmentCount} parameter greater than 0, then each element of the {@link VkRenderingInfo}{@code ::pColorAttachments} array with a {@code imageView} not equal to {@link VK10#VK_NULL_HANDLE NULL_HANDLE} <b>must</b> have been created with a sample count equal to the value of {@code rasterizationSamples} for the bound graphics pipeline</li>
     * <li>If the bound pipeline was created without a {@link VkAttachmentSampleCountInfoAMD} or {@link VkAttachmentSampleCountInfoNV} structure, and the <a href="https://registry.khronos.org/vulkan/specs/latest/html/vkspec.html#features-multisampledRenderToSingleSampled">{@code multisampledRenderToSingleSampled}</a> feature is not enabled, and {@link VkRenderingInfo}{@code ::pDepthAttachment→imageView} was not {@link VK10#VK_NULL_HANDLE NULL_HANDLE}, the value of {@code rasterizationSamples} for the bound graphics pipeline <b>must</b> be equal to the sample count used to create {@link VkRenderingInfo}{@code ::pDepthAttachment→imageView}</li>
     * <li>If the bound pipeline was created without a {@link VkAttachmentSampleCountInfoAMD} or {@link VkAttachmentSampleCountInfoNV} structure, and the <a href="https://registry.khronos.org/vulkan/specs/latest/html/vkspec.html#features-multisampledRenderToSingleSampled">{@code multisampledRenderToSingleSampled}</a> feature is not enabled, and {@link VkRenderingInfo}{@code ::pStencilAttachment→imageView} was not {@link VK10#VK_NULL_HANDLE NULL_HANDLE}, the value of {@code rasterizationSamples} for the bound graphics pipeline <b>must</b> be equal to the sample count used to create {@link VkRenderingInfo}{@code ::pStencilAttachment→imageView}</li>
     * <li>If this command has been called inside a render pass instance started with {@link VK13#vkCmdBeginRendering CmdBeginRendering}, and the {@code pNext} chain of {@link VkRenderingInfo} includes a {@link VkMultisampledRenderToSingleSampledInfoEXT} structure with {@code multisampledRenderToSingleSampledEnable} equal to {@link VK10#VK_TRUE TRUE}, then the value of {@code rasterizationSamples} for the bound graphics pipeline <b>must</b> be equal to {@link VkMultisampledRenderToSingleSampledInfoEXT}{@code ::rasterizationSamples}</li>
     * <li>If the current render pass instance was begun with {@link VK13#vkCmdBeginRendering CmdBeginRendering}, the bound pipeline <b>must</b> have been created with a {@link VkGraphicsPipelineCreateInfo}{@code ::renderPass} equal to {@link VK10#VK_NULL_HANDLE NULL_HANDLE}</li>
     * <li>If the current render pass instance was begun with {@link VK13#vkCmdBeginRendering CmdBeginRendering}, there is a graphics pipeline bound with a fragment shader that statically writes to a color attachment, the color write mask is not zero, color writes are enabled, and the corresponding element of the {@link VkRenderingInfo}{@code ::pColorAttachments→imageView} was not {@link VK10#VK_NULL_HANDLE NULL_HANDLE}, then the corresponding element of {@link VkPipelineRenderingCreateInfo}{@code ::pColorAttachmentFormats} used to create the pipeline <b>must</b> not be {@link VK10#VK_FORMAT_UNDEFINED FORMAT_UNDEFINED}</li>
     * <li>If the current render pass instance was begun with {@link VK13#vkCmdBeginRendering CmdBeginRendering}, there is a graphics pipeline bound, depth test is enabled, depth write is enabled, and the {@link VkRenderingInfo}{@code ::pDepthAttachment→imageView} was not {@link VK10#VK_NULL_HANDLE NULL_HANDLE}, then the {@link VkPipelineRenderingCreateInfo}{@code ::depthAttachmentFormat} used to create the pipeline <b>must</b> not be {@link VK10#VK_FORMAT_UNDEFINED FORMAT_UNDEFINED}</li>
     * <li>If the current render pass instance was begun with {@link VK13#vkCmdBeginRendering CmdBeginRendering}, there is a graphics pipeline bound, stencil test is enabled and the {@link VkRenderingInfo}{@code ::pStencilAttachment→imageView} was not {@link VK10#VK_NULL_HANDLE NULL_HANDLE}, then the {@link VkPipelineRenderingCreateInfo}{@code ::stencilAttachmentFormat} used to create the pipeline <b>must</b> not be {@link VK10#VK_FORMAT_UNDEFINED FORMAT_UNDEFINED}</li>
     * <li>If the <a href="https://registry.khronos.org/vulkan/specs/latest/html/vkspec.html#features-primitivesGeneratedQueryWithRasterizerDiscard">{@code primitivesGeneratedQueryWithRasterizerDiscard}</a> feature is not enabled and the {@link EXTPrimitivesGeneratedQuery#VK_QUERY_TYPE_PRIMITIVES_GENERATED_EXT QUERY_TYPE_PRIMITIVES_GENERATED_EXT} query is active, <a href="https://registry.khronos.org/vulkan/specs/latest/html/vkspec.html#primsrast-discard">rasterization discard</a> <b>must</b> not be enabled</li>
     * <li>If the <a href="https://registry.khronos.org/vulkan/specs/latest/html/vkspec.html#features-primitivesGeneratedQueryWithNonZeroStreams">{@code primitivesGeneratedQueryWithNonZeroStreams}</a> feature is not enabled and the {@link EXTPrimitivesGeneratedQuery#VK_QUERY_TYPE_PRIMITIVES_GENERATED_EXT QUERY_TYPE_PRIMITIVES_GENERATED_EXT} query is active, the bound graphics pipeline <b>must</b> not have been created with a non-zero value in {@link VkPipelineRasterizationStateStreamCreateInfoEXT}{@code ::rasterizationStream}</li>
     * <li>If the <a href="https://registry.khronos.org/vulkan/specs/latest/html/vkspec.html#features-depthClamp">{@code depthClamp}</a> feature is enabled, a shader object is bound to any graphics stage or a graphics pipeline is bound which was created with the {@link EXTExtendedDynamicState3#VK_DYNAMIC_STATE_DEPTH_CLAMP_ENABLE_EXT DYNAMIC_STATE_DEPTH_CLAMP_ENABLE_EXT} dynamic state enabled, and the <a href="https://registry.khronos.org/vulkan/specs/latest/html/vkspec.html#dynamic-state-current-value">current value</a> of {@code rasterizerDiscardEnable} is {@link VK10#VK_FALSE FALSE}, then {@link EXTShaderObject#vkCmdSetDepthClampEnableEXT CmdSetDepthClampEnableEXT} <b>must</b> have been called and not subsequently <a href="https://registry.khronos.org/vulkan/specs/latest/html/vkspec.html#dynamic-state-lifetime">invalidated</a> in the current command buffer prior to this drawing command</li>
     * <li>If a shader object is bound to any graphics stage or a graphics pipeline is bound which was created with the {@link EXTExtendedDynamicState3#VK_DYNAMIC_STATE_POLYGON_MODE_EXT DYNAMIC_STATE_POLYGON_MODE_EXT} dynamic state enabled, and the <a href="https://registry.khronos.org/vulkan/specs/latest/html/vkspec.html#dynamic-state-current-value">current value</a> of {@code rasterizerDiscardEnable} is {@link VK10#VK_FALSE FALSE}, then {@link EXTShaderObject#vkCmdSetPolygonModeEXT CmdSetPolygonModeEXT} <b>must</b> have been called and not subsequently <a href="https://registry.khronos.org/vulkan/specs/latest/html/vkspec.html#dynamic-state-lifetime">invalidated</a> in the current command buffer prior to this drawing command</li>
     * <li>If a shader object is bound to any graphics stage or a graphics pipeline is bound which was created with the {@link EXTExtendedDynamicState3#VK_DYNAMIC_STATE_RASTERIZATION_SAMPLES_EXT DYNAMIC_STATE_RASTERIZATION_SAMPLES_EXT} dynamic state enabled, and the <a href="https://registry.khronos.org/vulkan/specs/latest/html/vkspec.html#dynamic-state-current-value">current value</a> of {@code rasterizerDiscardEnable} is {@link VK10#VK_FALSE FALSE}, then {@link EXTShaderObject#vkCmdSetRasterizationSamplesEXT CmdSetRasterizationSamplesEXT} <b>must</b> have been called and not subsequently <a href="https://registry.khronos.org/vulkan/specs/latest/html/vkspec.html#dynamic-state-lifetime">invalidated</a> in the current command buffer prior to this drawing command</li>
     * <li>If a shader object is bound to any graphics stage or a graphics pipeline is bound which was created with the {@link EXTExtendedDynamicState3#VK_DYNAMIC_STATE_SAMPLE_MASK_EXT DYNAMIC_STATE_SAMPLE_MASK_EXT} dynamic state enabled, and the <a href="https://registry.khronos.org/vulkan/specs/latest/html/vkspec.html#dynamic-state-current-value">current value</a> of {@code rasterizerDiscardEnable} is {@link VK10#VK_FALSE FALSE}, then {@link EXTShaderObject#vkCmdSetSampleMaskEXT CmdSetSampleMaskEXT} <b>must</b> have been called and not subsequently <a href="https://registry.khronos.org/vulkan/specs/latest/html/vkspec.html#dynamic-state-lifetime">invalidated</a> in the current command buffer prior to this drawing command</li>
     * <li>If the bound graphics pipeline state was created with the {@link EXTExtendedDynamicState3#VK_DYNAMIC_STATE_ALPHA_TO_COVERAGE_ENABLE_EXT DYNAMIC_STATE_ALPHA_TO_COVERAGE_ENABLE_EXT} dynamic state enabled, and {@code alphaToCoverageEnable} was {@link VK10#VK_TRUE TRUE} in the last call to {@link EXTShaderObject#vkCmdSetAlphaToCoverageEnableEXT CmdSetAlphaToCoverageEnableEXT}, then the <a href="https://registry.khronos.org/vulkan/specs/latest/html/vkspec.html#interfaces-fragmentoutput">Fragment Output Interface</a> <b>must</b> contain a variable for the alpha {@code Component} word in {@code Location} 0 at {@code Index} 0</li>
     * <li>If a shader object is bound to any graphics stage, and the most recent call to {@link EXTShaderObject#vkCmdSetAlphaToCoverageEnableEXT CmdSetAlphaToCoverageEnableEXT} in the current command buffer set {@code alphaToCoverageEnable} to {@link VK10#VK_TRUE TRUE}, then the <a href="https://registry.khronos.org/vulkan/specs/latest/html/vkspec.html#interfaces-fragmentoutput">Fragment Output Interface</a> <b>must</b> contain a variable for the alpha {@code Component} word in {@code Location} 0 at {@code Index} 0</li>
     * <li>If a shader object is bound to any graphics stage or a graphics pipeline is bound which was created with the {@link EXTExtendedDynamicState3#VK_DYNAMIC_STATE_ALPHA_TO_COVERAGE_ENABLE_EXT DYNAMIC_STATE_ALPHA_TO_COVERAGE_ENABLE_EXT} dynamic state enabled, and the <a href="https://registry.khronos.org/vulkan/specs/latest/html/vkspec.html#dynamic-state-current-value">current value</a> of {@code rasterizerDiscardEnable} is {@link VK10#VK_FALSE FALSE}, then {@link EXTShaderObject#vkCmdSetAlphaToCoverageEnableEXT CmdSetAlphaToCoverageEnableEXT} <b>must</b> have been called and not subsequently <a href="https://registry.khronos.org/vulkan/specs/latest/html/vkspec.html#dynamic-state-lifetime">invalidated</a> in the current command buffer prior to this drawing command</li>
     * <li>If the <a href="https://registry.khronos.org/vulkan/specs/latest/html/vkspec.html#features-alphaToOne">{@code alphaToOne}</a> feature is enabled, a shader object is bound to any graphics stage or a graphics pipeline is bound which was created with the {@link EXTExtendedDynamicState3#VK_DYNAMIC_STATE_ALPHA_TO_ONE_ENABLE_EXT DYNAMIC_STATE_ALPHA_TO_ONE_ENABLE_EXT} dynamic state enabled, and the <a href="https://registry.khronos.org/vulkan/specs/latest/html/vkspec.html#dynamic-state-current-value">current value</a> of {@code rasterizerDiscardEnable} is {@link VK10#VK_FALSE FALSE}, then {@link EXTShaderObject#vkCmdSetAlphaToOneEnableEXT CmdSetAlphaToOneEnableEXT} <b>must</b> have been called and not subsequently <a href="https://registry.khronos.org/vulkan/specs/latest/html/vkspec.html#dynamic-state-lifetime">invalidated</a> in the current command buffer prior to this drawing command</li>
     * <li>If the <a href="https://registry.khronos.org/vulkan/specs/latest/html/vkspec.html#features-logicOp">{@code logicOp}</a> feature is enabled, a shader object is bound to the {@link VK10#VK_SHADER_STAGE_FRAGMENT_BIT SHADER_STAGE_FRAGMENT_BIT} stage or a graphics pipeline is bound which was created with the {@link EXTExtendedDynamicState3#VK_DYNAMIC_STATE_LOGIC_OP_ENABLE_EXT DYNAMIC_STATE_LOGIC_OP_ENABLE_EXT} dynamic state enabled, and the <a href="https://registry.khronos.org/vulkan/specs/latest/html/vkspec.html#dynamic-state-current-value">current value</a> of {@code rasterizerDiscardEnable} is {@link VK10#VK_FALSE FALSE}, then {@link EXTShaderObject#vkCmdSetLogicOpEnableEXT CmdSetLogicOpEnableEXT} <b>must</b> have been called and not subsequently <a href="https://registry.khronos.org/vulkan/specs/latest/html/vkspec.html#dynamic-state-lifetime">invalidated</a> in the current command buffer prior to this drawing command</li>
     * <li>If the bound graphics pipeline state was created with the {@link EXTExtendedDynamicState3#VK_DYNAMIC_STATE_COLOR_BLEND_ENABLE_EXT DYNAMIC_STATE_COLOR_BLEND_ENABLE_EXT} dynamic state enabled then {@link EXTShaderObject#vkCmdSetColorBlendEnableEXT CmdSetColorBlendEnableEXT} <b>must</b> have been called and not subsequently <a href="https://registry.khronos.org/vulkan/specs/latest/html/vkspec.html#dynamic-state-lifetime">invalidated</a> in the current command buffer prior to this drawing command</li>
     * <li>If a shader object is bound to the {@link VK10#VK_SHADER_STAGE_FRAGMENT_BIT SHADER_STAGE_FRAGMENT_BIT} stage, and both the most recent call to {@link VK13#vkCmdSetRasterizerDiscardEnable CmdSetRasterizerDiscardEnable} in the current command buffer set {@code rasterizerDiscardEnable} to {@link VK10#VK_FALSE FALSE} and there are color attachments bound, then {@link EXTShaderObject#vkCmdSetColorBlendEnableEXT CmdSetColorBlendEnableEXT} <b>must</b> have been called and not subsequently <a href="https://registry.khronos.org/vulkan/specs/latest/html/vkspec.html#dynamic-state-lifetime">invalidated</a> in the current command buffer prior to this drawing command</li>
     * <li>If the bound graphics pipeline state was created with the {@link EXTExtendedDynamicState3#VK_DYNAMIC_STATE_COLOR_BLEND_EQUATION_EXT DYNAMIC_STATE_COLOR_BLEND_EQUATION_EXT} dynamic state enabled then {@link EXTShaderObject#vkCmdSetColorBlendEquationEXT CmdSetColorBlendEquationEXT} <b>must</b> have been called and not subsequently <a href="https://registry.khronos.org/vulkan/specs/latest/html/vkspec.html#dynamic-state-lifetime">invalidated</a> in the current command buffer prior to this drawing command</li>
     * <li>If a shader object is bound to the {@link VK10#VK_SHADER_STAGE_FRAGMENT_BIT SHADER_STAGE_FRAGMENT_BIT} stage, and the most recent call to {@link VK13#vkCmdSetRasterizerDiscardEnable CmdSetRasterizerDiscardEnable} in the current command buffer set {@code rasterizerDiscardEnable} to {@link VK10#VK_FALSE FALSE}, and the most recent call to {@link EXTShaderObject#vkCmdSetColorBlendEnableEXT CmdSetColorBlendEnableEXT} for any attachment set that attachment’s value in {@code pColorBlendEnables} to {@link VK10#VK_TRUE TRUE}, then {@link EXTShaderObject#vkCmdSetColorBlendEquationEXT CmdSetColorBlendEquationEXT} <b>must</b> have been called and not subsequently <a href="https://registry.khronos.org/vulkan/specs/latest/html/vkspec.html#dynamic-state-lifetime">invalidated</a> in the current command buffer prior to this drawing command</li>
     * <li>If the bound graphics pipeline state was created with the {@link EXTExtendedDynamicState3#VK_DYNAMIC_STATE_COLOR_WRITE_MASK_EXT DYNAMIC_STATE_COLOR_WRITE_MASK_EXT} dynamic state enabled then {@link EXTShaderObject#vkCmdSetColorWriteMaskEXT CmdSetColorWriteMaskEXT} <b>must</b> have been called and not subsequently <a href="https://registry.khronos.org/vulkan/specs/latest/html/vkspec.html#dynamic-state-lifetime">invalidated</a> in the current command buffer prior to this drawing command</li>
     * <li>If a shader object is bound to the {@link VK10#VK_SHADER_STAGE_FRAGMENT_BIT SHADER_STAGE_FRAGMENT_BIT} stage, and both the most recent call to {@link VK13#vkCmdSetRasterizerDiscardEnable CmdSetRasterizerDiscardEnable} in the current command buffer set {@code rasterizerDiscardEnable} to {@link VK10#VK_FALSE FALSE} and there are color attachments bound, then {@link EXTShaderObject#vkCmdSetColorWriteMaskEXT CmdSetColorWriteMaskEXT} <b>must</b> have been called and not subsequently <a href="https://registry.khronos.org/vulkan/specs/latest/html/vkspec.html#dynamic-state-lifetime">invalidated</a> in the current command buffer prior to this drawing command</li>
     * <li>If the <a href="https://registry.khronos.org/vulkan/specs/latest/html/vkspec.html#features-geometryStreams">{@code geometryStreams}</a> feature is enabled, and a shader object is bound to the {@link VK10#VK_SHADER_STAGE_GEOMETRY_BIT SHADER_STAGE_GEOMETRY_BIT} stage or a graphics pipeline is bound which was created with the {@link EXTExtendedDynamicState3#VK_DYNAMIC_STATE_RASTERIZATION_STREAM_EXT DYNAMIC_STATE_RASTERIZATION_STREAM_EXT} dynamic state enabled, then {@link EXTShaderObject#vkCmdSetRasterizationStreamEXT CmdSetRasterizationStreamEXT} <b>must</b> have been called and not subsequently <a href="https://registry.khronos.org/vulkan/specs/latest/html/vkspec.html#dynamic-state-lifetime">invalidated</a> in the current command buffer prior to this drawing command</li>
     * <li>If the {@link EXTConservativeRasterization VK_EXT_conservative_rasterization} extension is enabled, a shader object is bound to any graphics stage or a graphics pipeline is bound which was created with the {@link EXTExtendedDynamicState3#VK_DYNAMIC_STATE_CONSERVATIVE_RASTERIZATION_MODE_EXT DYNAMIC_STATE_CONSERVATIVE_RASTERIZATION_MODE_EXT} dynamic state enabled, and the <a href="https://registry.khronos.org/vulkan/specs/latest/html/vkspec.html#dynamic-state-current-value">current value</a> of {@code rasterizerDiscardEnable} is {@link VK10#VK_FALSE FALSE}, then {@link EXTShaderObject#vkCmdSetConservativeRasterizationModeEXT CmdSetConservativeRasterizationModeEXT} <b>must</b> have been called and not subsequently <a href="https://registry.khronos.org/vulkan/specs/latest/html/vkspec.html#dynamic-state-lifetime">invalidated</a> in the current command buffer prior to this drawing command</li>
     * <li>If the {@link EXTConservativeRasterization VK_EXT_conservative_rasterization} extension is enabled, a shader object is bound to any graphics stage or a graphics pipeline is bound which was created with the {@link EXTExtendedDynamicState3#VK_DYNAMIC_STATE_EXTRA_PRIMITIVE_OVERESTIMATION_SIZE_EXT DYNAMIC_STATE_EXTRA_PRIMITIVE_OVERESTIMATION_SIZE_EXT} dynamic state enabled, the <a href="https://registry.khronos.org/vulkan/specs/latest/html/vkspec.html#dynamic-state-current-value">current value</a> of {@code rasterizerDiscardEnable} is {@link VK10#VK_FALSE FALSE}, and the <a href="https://registry.khronos.org/vulkan/specs/latest/html/vkspec.html#dynamic-state-current-value">current value</a> of {@code conservativeRasterizationMode} is {@link EXTConservativeRasterization#VK_CONSERVATIVE_RASTERIZATION_MODE_OVERESTIMATE_EXT CONSERVATIVE_RASTERIZATION_MODE_OVERESTIMATE_EXT}, then {@link EXTShaderObject#vkCmdSetExtraPrimitiveOverestimationSizeEXT CmdSetExtraPrimitiveOverestimationSizeEXT} <b>must</b> have been called and not subsequently <a href="https://registry.khronos.org/vulkan/specs/latest/html/vkspec.html#dynamic-state-lifetime">invalidated</a> in the current command buffer prior to this drawing command</li>
     * <li>If the <a href="https://registry.khronos.org/vulkan/specs/latest/html/vkspec.html#features-depthClipEnable">{@code depthClipEnable}</a> feature is enabled, and a shader object is bound to any graphics stage or a graphics pipeline is bound which was created with the {@link EXTExtendedDynamicState3#VK_DYNAMIC_STATE_DEPTH_CLIP_ENABLE_EXT DYNAMIC_STATE_DEPTH_CLIP_ENABLE_EXT} dynamic state, then {@link EXTShaderObject#vkCmdSetDepthClipEnableEXT CmdSetDepthClipEnableEXT} <b>must</b> have been called and not subsequently <a href="https://registry.khronos.org/vulkan/specs/latest/html/vkspec.html#dynamic-state-lifetime">invalidated</a> in the current command buffer prior to this drawing command</li>
     * <li>If the bound graphics pipeline state was created with the {@link EXTExtendedDynamicState3#VK_DYNAMIC_STATE_COLOR_BLEND_ADVANCED_EXT DYNAMIC_STATE_COLOR_BLEND_ADVANCED_EXT} dynamic state enabled then {@link EXTShaderObject#vkCmdSetColorBlendAdvancedEXT CmdSetColorBlendAdvancedEXT} <b>must</b> have been called and not subsequently <a href="https://registry.khronos.org/vulkan/specs/latest/html/vkspec.html#dynamic-state-lifetime">invalidated</a> in the current command buffer prior to this drawing command</li>
     * <li>If the {@link EXTBlendOperationAdvanced VK_EXT_blend_operation_advanced} extension is enabled, and a shader object is bound to the {@link VK10#VK_SHADER_STAGE_FRAGMENT_BIT SHADER_STAGE_FRAGMENT_BIT} stage, and the most recent call to {@link VK13#vkCmdSetRasterizerDiscardEnable CmdSetRasterizerDiscardEnable} in the current command buffer set {@code rasterizerDiscardEnable} to {@link VK10#VK_FALSE FALSE}, then at least one of {@link EXTShaderObject#vkCmdSetColorBlendEquationEXT CmdSetColorBlendEquationEXT} and {@link EXTShaderObject#vkCmdSetColorBlendAdvancedEXT CmdSetColorBlendAdvancedEXT} <b>must</b> have been called and not subsequently <a href="https://registry.khronos.org/vulkan/specs/latest/html/vkspec.html#dynamic-state-lifetime">invalidated</a> in the current command buffer prior to this drawing command</li>
     * <li>If the {@link EXTProvokingVertex VK_EXT_provoking_vertex} extension is enabled, a shader object is bound to the {@link VK10#VK_SHADER_STAGE_VERTEX_BIT SHADER_STAGE_VERTEX_BIT} stage or a graphics pipeline is bound which was created with the {@link EXTExtendedDynamicState3#VK_DYNAMIC_STATE_PROVOKING_VERTEX_MODE_EXT DYNAMIC_STATE_PROVOKING_VERTEX_MODE_EXT} dynamic state enabled, and the <a href="https://registry.khronos.org/vulkan/specs/latest/html/vkspec.html#dynamic-state-current-value">current value</a> of {@code rasterizerDiscardEnable} is {@link VK10#VK_FALSE FALSE}, then {@link EXTShaderObject#vkCmdSetProvokingVertexModeEXT CmdSetProvokingVertexModeEXT} <b>must</b> have been called and not subsequently <a href="https://registry.khronos.org/vulkan/specs/latest/html/vkspec.html#dynamic-state-lifetime">invalidated</a> in the current command buffer prior to this drawing command</li>
     * <li>If the bound graphics pipeline state was created with the {@link EXTExtendedDynamicState3#VK_DYNAMIC_STATE_LINE_RASTERIZATION_MODE_EXT DYNAMIC_STATE_LINE_RASTERIZATION_MODE_EXT} dynamic state enabled then {@link EXTShaderObject#vkCmdSetLineRasterizationModeEXT CmdSetLineRasterizationModeEXT} <b>must</b> have been called and not subsequently <a href="https://registry.khronos.org/vulkan/specs/latest/html/vkspec.html#dynamic-state-lifetime">invalidated</a> in the current command buffer prior to this drawing command</li>
     * <li>If any of the <a href="https://registry.khronos.org/vulkan/specs/latest/html/vkspec.html#features-stippledRectangularLines">{@code stippledRectangularLines}</a>, <a href="https://registry.khronos.org/vulkan/specs/latest/html/vkspec.html#features-stippledBresenhamLines">{@code stippledBresenhamLines}</a> or <a href="https://registry.khronos.org/vulkan/specs/latest/html/vkspec.html#features-stippledSmoothLines">{@code stippledSmoothLines}</a> features are enabled, and a shader object is bound to any graphics stage, and the most recent call to {@link VK13#vkCmdSetRasterizerDiscardEnable CmdSetRasterizerDiscardEnable} in the current command buffer set {@code rasterizerDiscardEnable} to {@link VK10#VK_FALSE FALSE}, and the most recent call to {@link EXTShaderObject#vkCmdSetPolygonModeEXT CmdSetPolygonModeEXT} in the current command buffer set {@code polygonMode} to {@link VK10#VK_POLYGON_MODE_LINE POLYGON_MODE_LINE}, then {@link EXTShaderObject#vkCmdSetLineRasterizationModeEXT CmdSetLineRasterizationModeEXT} <b>must</b> have been called and not subsequently <a href="https://registry.khronos.org/vulkan/specs/latest/html/vkspec.html#dynamic-state-lifetime">invalidated</a> in the current command buffer prior to this drawing command</li>
     * <li>If any of the <a href="https://registry.khronos.org/vulkan/specs/latest/html/vkspec.html#features-stippledRectangularLines">{@code stippledRectangularLines}</a>, <a href="https://registry.khronos.org/vulkan/specs/latest/html/vkspec.html#features-stippledBresenhamLines">{@code stippledBresenhamLines}</a> or <a href="https://registry.khronos.org/vulkan/specs/latest/html/vkspec.html#features-stippledSmoothLines">{@code stippledSmoothLines}</a> features are enabled, and a shader object is bound to the {@link VK10#VK_SHADER_STAGE_VERTEX_BIT SHADER_STAGE_VERTEX_BIT} stage, and the most recent call to {@link VK13#vkCmdSetRasterizerDiscardEnable CmdSetRasterizerDiscardEnable} in the current command buffer set {@code rasterizerDiscardEnable} to {@link VK10#VK_FALSE FALSE}, and the most recent call to {@link VK13#vkCmdSetPrimitiveTopology CmdSetPrimitiveTopology} in the current command buffer set {@code primitiveTopology} to any line topology, then {@link EXTShaderObject#vkCmdSetLineRasterizationModeEXT CmdSetLineRasterizationModeEXT} <b>must</b> have been called and not subsequently <a href="https://registry.khronos.org/vulkan/specs/latest/html/vkspec.html#dynamic-state-lifetime">invalidated</a> in the current command buffer prior to this drawing command</li>
     * <li>If any of the <a href="https://registry.khronos.org/vulkan/specs/latest/html/vkspec.html#features-stippledRectangularLines">{@code stippledRectangularLines}</a>, <a href="https://registry.khronos.org/vulkan/specs/latest/html/vkspec.html#features-stippledBresenhamLines">{@code stippledBresenhamLines}</a> or <a href="https://registry.khronos.org/vulkan/specs/latest/html/vkspec.html#features-stippledSmoothLines">{@code stippledSmoothLines}</a> features are enabled, and a shader object that outputs line primitives is bound to the {@link VK10#VK_SHADER_STAGE_TESSELLATION_EVALUATION_BIT SHADER_STAGE_TESSELLATION_EVALUATION_BIT} or {@link VK10#VK_SHADER_STAGE_GEOMETRY_BIT SHADER_STAGE_GEOMETRY_BIT} stage, and the most recent call to {@link VK13#vkCmdSetRasterizerDiscardEnable CmdSetRasterizerDiscardEnable} in the current command buffer set {@code rasterizerDiscardEnable} to {@link VK10#VK_FALSE FALSE}, then {@link EXTShaderObject#vkCmdSetLineRasterizationModeEXT CmdSetLineRasterizationModeEXT} <b>must</b> have been called and not subsequently <a href="https://registry.khronos.org/vulkan/specs/latest/html/vkspec.html#dynamic-state-lifetime">invalidated</a> in the current command buffer prior to this drawing command</li>
     * <li>If the bound graphics pipeline state was created with the {@link EXTExtendedDynamicState3#VK_DYNAMIC_STATE_LINE_STIPPLE_ENABLE_EXT DYNAMIC_STATE_LINE_STIPPLE_ENABLE_EXT} dynamic state enabled then {@link EXTShaderObject#vkCmdSetLineStippleEnableEXT CmdSetLineStippleEnableEXT} <b>must</b> have been called and not subsequently <a href="https://registry.khronos.org/vulkan/specs/latest/html/vkspec.html#dynamic-state-lifetime">invalidated</a> in the current command buffer prior to this drawing command</li>
     * <li>If any of the <a href="https://registry.khronos.org/vulkan/specs/latest/html/vkspec.html#features-stippledRectangularLines">{@code stippledRectangularLines}</a>, <a href="https://registry.khronos.org/vulkan/specs/latest/html/vkspec.html#features-stippledBresenhamLines">{@code stippledBresenhamLines}</a> or <a href="https://registry.khronos.org/vulkan/specs/latest/html/vkspec.html#features-stippledSmoothLines">{@code stippledSmoothLines}</a> features are enabled, and a shader object is bound to any graphics stage, and the most recent call to {@link VK13#vkCmdSetRasterizerDiscardEnable CmdSetRasterizerDiscardEnable} in the current command buffer set {@code rasterizerDiscardEnable} to {@link VK10#VK_FALSE FALSE}, and the most recent call to {@link EXTShaderObject#vkCmdSetPolygonModeEXT CmdSetPolygonModeEXT} in the current command buffer set {@code polygonMode} to {@link VK10#VK_POLYGON_MODE_LINE POLYGON_MODE_LINE}, then {@link EXTShaderObject#vkCmdSetLineStippleEnableEXT CmdSetLineStippleEnableEXT} <b>must</b> have been called and not subsequently <a href="https://registry.khronos.org/vulkan/specs/latest/html/vkspec.html#dynamic-state-lifetime">invalidated</a> in the current command buffer prior to this drawing command</li>
     * <li>If any of the <a href="https://registry.khronos.org/vulkan/specs/latest/html/vkspec.html#features-stippledRectangularLines">{@code stippledRectangularLines}</a>, <a href="https://registry.khronos.org/vulkan/specs/latest/html/vkspec.html#features-stippledBresenhamLines">{@code stippledBresenhamLines}</a> or <a href="https://registry.khronos.org/vulkan/specs/latest/html/vkspec.html#features-stippledSmoothLines">{@code stippledSmoothLines}</a> features are enabled, and a shader object is bound to the {@link VK10#VK_SHADER_STAGE_VERTEX_BIT SHADER_STAGE_VERTEX_BIT} stage, and the most recent call to {@link VK13#vkCmdSetRasterizerDiscardEnable CmdSetRasterizerDiscardEnable} in the current command buffer set {@code rasterizerDiscardEnable} to {@link VK10#VK_FALSE FALSE}, and the most recent call to {@link VK13#vkCmdSetPrimitiveTopology CmdSetPrimitiveTopology} in the current command buffer set {@code primitiveTopology} to any line topology, then {@link EXTShaderObject#vkCmdSetLineStippleEnableEXT CmdSetLineStippleEnableEXT} <b>must</b> have been called and not subsequently <a href="https://registry.khronos.org/vulkan/specs/latest/html/vkspec.html#dynamic-state-lifetime">invalidated</a> in the current command buffer prior to this drawing command</li>
     * <li>If any of the <a href="https://registry.khronos.org/vulkan/specs/latest/html/vkspec.html#features-stippledRectangularLines">{@code stippledRectangularLines}</a>, <a href="https://registry.khronos.org/vulkan/specs/latest/html/vkspec.html#features-stippledBresenhamLines">{@code stippledBresenhamLines}</a> or <a href="https://registry.khronos.org/vulkan/specs/latest/html/vkspec.html#features-stippledSmoothLines">{@code stippledSmoothLines}</a> features are enabled, and a shader object that outputs line primitives is bound to the {@link VK10#VK_SHADER_STAGE_TESSELLATION_EVALUATION_BIT SHADER_STAGE_TESSELLATION_EVALUATION_BIT} or {@link VK10#VK_SHADER_STAGE_GEOMETRY_BIT SHADER_STAGE_GEOMETRY_BIT} stage, and the most recent call to {@link VK13#vkCmdSetRasterizerDiscardEnable CmdSetRasterizerDiscardEnable} in the current command buffer set {@code rasterizerDiscardEnable} to {@link VK10#VK_FALSE FALSE}, then {@link EXTShaderObject#vkCmdSetLineStippleEnableEXT CmdSetLineStippleEnableEXT} <b>must</b> have been called and not subsequently <a href="https://registry.khronos.org/vulkan/specs/latest/html/vkspec.html#dynamic-state-lifetime">invalidated</a> in the current command buffer prior to this drawing command</li>
     * <li>If any of the <a href="https://registry.khronos.org/vulkan/specs/latest/html/vkspec.html#features-stippledRectangularLines">{@code stippledRectangularLines}</a>, <a href="https://registry.khronos.org/vulkan/specs/latest/html/vkspec.html#features-stippledBresenhamLines">{@code stippledBresenhamLines}</a> or <a href="https://registry.khronos.org/vulkan/specs/latest/html/vkspec.html#features-stippledSmoothLines">{@code stippledSmoothLines}</a> features are enabled and a shader object is bound to any graphics stage, or a bound graphics pipeline state was created with the {@link VK14#VK_DYNAMIC_STATE_LINE_STIPPLE DYNAMIC_STATE_LINE_STIPPLE} dynamic state enabled, the <a href="https://registry.khronos.org/vulkan/specs/latest/html/vkspec.html#dynamic-state-current-value">current value</a> of {@code rasterizerDiscardEnable} is {@link VK10#VK_FALSE FALSE}, and the <a href="https://registry.khronos.org/vulkan/specs/latest/html/vkspec.html#dynamic-state-current-value">current value</a> of {@code stippledLineEnable} is {@link VK10#VK_TRUE TRUE}, then {@link VK14#vkCmdSetLineStipple CmdSetLineStipple} <b>must</b> have been called and not subsequently <a href="https://registry.khronos.org/vulkan/specs/latest/html/vkspec.html#dynamic-state-lifetime">invalidated</a> in the current command buffer prior to this drawing command</li>
     * <li>If the <a href="https://registry.khronos.org/vulkan/specs/latest/html/vkspec.html#features-depthClipControl">{@code depthClipControl}</a> feature is enabled, and a shader object is bound to any graphics stage or a graphics pipeline is bound which was created with the {@link EXTExtendedDynamicState3#VK_DYNAMIC_STATE_DEPTH_CLIP_NEGATIVE_ONE_TO_ONE_EXT DYNAMIC_STATE_DEPTH_CLIP_NEGATIVE_ONE_TO_ONE_EXT} dynamic state enabled, then {@link EXTShaderObject#vkCmdSetDepthClipNegativeOneToOneEXT CmdSetDepthClipNegativeOneToOneEXT} <b>must</b> have been called and not subsequently <a href="https://registry.khronos.org/vulkan/specs/latest/html/vkspec.html#dynamic-state-lifetime">invalidated</a> in the current command buffer prior to this drawing command</li>
     * <li>If the <a href="https://registry.khronos.org/vulkan/specs/latest/html/vkspec.html#features-depthClampControl">{@code depthClampControl}</a> feature is enabled, and a shader object is bound to any graphics stage or a graphics pipeline is bound which was created with the {@link EXTDepthClampControl#VK_DYNAMIC_STATE_DEPTH_CLAMP_RANGE_EXT DYNAMIC_STATE_DEPTH_CLAMP_RANGE_EXT} dynamic state enabled, and the <a href="https://registry.khronos.org/vulkan/specs/latest/html/vkspec.html#dynamic-state-current-value">current value</a> of {@code depthClampEnable} is {@link VK10#VK_TRUE TRUE}, then {@link EXTShaderObject#vkCmdSetDepthClampRangeEXT CmdSetDepthClampRangeEXT} <b>must</b> have been called and not subsequently <a href="https://registry.khronos.org/vulkan/specs/latest/html/vkspec.html#dynamic-state-lifetime">invalidated</a> in the current command buffer prior to this drawing command</li>
     * <li>If the {@link NVClipSpaceWScaling VK_NV_clip_space_w_scaling} extension is enabled, and a shader object is bound to any graphics stage or a graphics pipeline is bound which was created with the {@link EXTExtendedDynamicState3#VK_DYNAMIC_STATE_VIEWPORT_W_SCALING_ENABLE_NV DYNAMIC_STATE_VIEWPORT_W_SCALING_ENABLE_NV} dynamic state enabled, then {@link EXTShaderObject#vkCmdSetViewportWScalingEnableNV CmdSetViewportWScalingEnableNV} <b>must</b> have been called and not subsequently <a href="https://registry.khronos.org/vulkan/specs/latest/html/vkspec.html#dynamic-state-lifetime">invalidated</a> in the current command buffer prior to this drawing command</li>
     * <li>If the {@link NVViewportSwizzle VK_NV_viewport_swizzle} extension is enabled, and a shader object is bound to any graphics stage or a graphics pipeline is bound which was created with the {@link EXTExtendedDynamicState3#VK_DYNAMIC_STATE_VIEWPORT_SWIZZLE_NV DYNAMIC_STATE_VIEWPORT_SWIZZLE_NV} dynamic state enabled, then {@link EXTShaderObject#vkCmdSetViewportSwizzleNV CmdSetViewportSwizzleNV} <b>must</b> have been called and not subsequently <a href="https://registry.khronos.org/vulkan/specs/latest/html/vkspec.html#dynamic-state-lifetime">invalidated</a> in the current command buffer prior to this drawing command</li>
     * <li>If the {@link NVFragmentCoverageToColor VK_NV_fragment_coverage_to_color} extension is enabled, a shader object is bound to the {@link VK10#VK_SHADER_STAGE_FRAGMENT_BIT SHADER_STAGE_FRAGMENT_BIT} stage or a graphics pipeline is bound which was created with the {@link EXTExtendedDynamicState3#VK_DYNAMIC_STATE_COVERAGE_TO_COLOR_ENABLE_NV DYNAMIC_STATE_COVERAGE_TO_COLOR_ENABLE_NV} dynamic state enabled, and the <a href="https://registry.khronos.org/vulkan/specs/latest/html/vkspec.html#dynamic-state-current-value">current value</a> of {@code rasterizerDiscardEnable} is {@link VK10#VK_FALSE FALSE}, then {@link EXTShaderObject#vkCmdSetCoverageToColorEnableNV CmdSetCoverageToColorEnableNV} <b>must</b> have been called and not subsequently <a href="https://registry.khronos.org/vulkan/specs/latest/html/vkspec.html#dynamic-state-lifetime">invalidated</a> in the current command buffer prior to this drawing command</li>
     * <li>If the {@link NVFragmentCoverageToColor VK_NV_fragment_coverage_to_color} extension is enabled, a shader object is bound to the {@link VK10#VK_SHADER_STAGE_FRAGMENT_BIT SHADER_STAGE_FRAGMENT_BIT} stage or a graphics pipeline is bound which was created with the {@link EXTExtendedDynamicState3#VK_DYNAMIC_STATE_COVERAGE_TO_COLOR_LOCATION_NV DYNAMIC_STATE_COVERAGE_TO_COLOR_LOCATION_NV} dynamic state enabled, the <a href="https://registry.khronos.org/vulkan/specs/latest/html/vkspec.html#dynamic-state-current-value">current value</a> of {@code rasterizerDiscardEnable} is {@link VK10#VK_FALSE FALSE}, and the <a href="https://registry.khronos.org/vulkan/specs/latest/html/vkspec.html#dynamic-state-current-value">current value</a> of {@code coverageToColorEnable} is {@link VK10#VK_TRUE TRUE}, then {@link EXTShaderObject#vkCmdSetCoverageToColorLocationNV CmdSetCoverageToColorLocationNV} <b>must</b> have been called and not subsequently <a href="https://registry.khronos.org/vulkan/specs/latest/html/vkspec.html#dynamic-state-lifetime">invalidated</a> in the current command buffer prior to this drawing command</li>
     * <li>If the {@link NVFramebufferMixedSamples VK_NV_framebuffer_mixed_samples} extension is enabled, a shader object is bound to any graphics stage or a graphics pipeline is bound which was created with the {@link EXTExtendedDynamicState3#VK_DYNAMIC_STATE_COVERAGE_MODULATION_MODE_NV DYNAMIC_STATE_COVERAGE_MODULATION_MODE_NV} dynamic state enabled, and the <a href="https://registry.khronos.org/vulkan/specs/latest/html/vkspec.html#dynamic-state-current-value">current value</a> of {@code rasterizerDiscardEnable} is {@link VK10#VK_FALSE FALSE}, then {@link EXTShaderObject#vkCmdSetCoverageModulationModeNV CmdSetCoverageModulationModeNV} <b>must</b> have been called and not subsequently <a href="https://registry.khronos.org/vulkan/specs/latest/html/vkspec.html#dynamic-state-lifetime">invalidated</a> in the current command buffer prior to this drawing command</li>
     * <li>If the {@link NVFramebufferMixedSamples VK_NV_framebuffer_mixed_samples} extension is enabled, a shader object is bound to any graphics stage or a graphics pipeline is bound which was created with the {@link EXTExtendedDynamicState3#VK_DYNAMIC_STATE_COVERAGE_MODULATION_TABLE_ENABLE_NV DYNAMIC_STATE_COVERAGE_MODULATION_TABLE_ENABLE_NV} dynamic state enabled, the <a href="https://registry.khronos.org/vulkan/specs/latest/html/vkspec.html#dynamic-state-current-value">current value</a> of {@code rasterizerDiscardEnable} is {@link VK10#VK_FALSE FALSE}, and the <a href="https://registry.khronos.org/vulkan/specs/latest/html/vkspec.html#dynamic-state-current-value">current value</a> of {@code coverageModulationMode} is any value other than {@link NVFramebufferMixedSamples#VK_COVERAGE_MODULATION_MODE_NONE_NV COVERAGE_MODULATION_MODE_NONE_NV}, then {@link EXTShaderObject#vkCmdSetCoverageModulationTableEnableNV CmdSetCoverageModulationTableEnableNV} <b>must</b> have been called and not subsequently <a href="https://registry.khronos.org/vulkan/specs/latest/html/vkspec.html#dynamic-state-lifetime">invalidated</a> in the current command buffer prior to this drawing command</li>
     * <li>If the {@link NVFramebufferMixedSamples VK_NV_framebuffer_mixed_samples} extension is enabled, a shader object is bound to any graphics stage or a graphics pipeline is bound which was created with the {@link EXTExtendedDynamicState3#VK_DYNAMIC_STATE_COVERAGE_MODULATION_TABLE_NV DYNAMIC_STATE_COVERAGE_MODULATION_TABLE_NV} dynamic state enabled, the <a href="https://registry.khronos.org/vulkan/specs/latest/html/vkspec.html#dynamic-state-current-value">current value</a> of {@code rasterizerDiscardEnable} is {@link VK10#VK_FALSE FALSE}, and the <a href="https://registry.khronos.org/vulkan/specs/latest/html/vkspec.html#dynamic-state-current-value">current value</a> of {@code coverageModulationTableEnable} is {@link VK10#VK_TRUE TRUE}, then {@link EXTShaderObject#vkCmdSetCoverageModulationTableNV CmdSetCoverageModulationTableNV} <b>must</b> have been called and not subsequently <a href="https://registry.khronos.org/vulkan/specs/latest/html/vkspec.html#dynamic-state-lifetime">invalidated</a> in the current command buffer prior to this drawing command</li>
     * <li>If the <a href="https://registry.khronos.org/vulkan/specs/latest/html/vkspec.html#features-shadingRateImage">{@code shadingRateImage}</a> feature is enabled, a shader object is bound to any graphics stage or a graphics pipeline is bound which was created with the {@link EXTExtendedDynamicState3#VK_DYNAMIC_STATE_SHADING_RATE_IMAGE_ENABLE_NV DYNAMIC_STATE_SHADING_RATE_IMAGE_ENABLE_NV} dynamic state enabled, and the <a href="https://registry.khronos.org/vulkan/specs/latest/html/vkspec.html#dynamic-state-current-value">current value</a> of {@code rasterizerDiscardEnable} is {@link VK10#VK_FALSE FALSE}, then {@link EXTShaderObject#vkCmdSetShadingRateImageEnableNV CmdSetShadingRateImageEnableNV} <b>must</b> have been called and not subsequently <a href="https://registry.khronos.org/vulkan/specs/latest/html/vkspec.html#dynamic-state-lifetime">invalidated</a> in the current command buffer prior to this drawing command</li>
     * <li>If the <a href="https://registry.khronos.org/vulkan/specs/latest/html/vkspec.html#features-pipelineFragmentShadingRate">{@code pipelineFragmentShadingRate}</a> feature is enabled, a shader object is bound to the {@link VK10#VK_SHADER_STAGE_FRAGMENT_BIT SHADER_STAGE_FRAGMENT_BIT} stage or a graphics pipeline is bound which was created with the {@link KHRFragmentShadingRate#VK_DYNAMIC_STATE_FRAGMENT_SHADING_RATE_KHR DYNAMIC_STATE_FRAGMENT_SHADING_RATE_KHR} dynamic state enabled, and the <a href="https://registry.khronos.org/vulkan/specs/latest/html/vkspec.html#dynamic-state-current-value">current value</a> of {@code rasterizerDiscardEnable} is {@link VK10#VK_FALSE FALSE}, then {@link KHRFragmentShadingRate#vkCmdSetFragmentShadingRateKHR CmdSetFragmentShadingRateKHR} <b>must</b> have been called and not subsequently <a href="https://registry.khronos.org/vulkan/specs/latest/html/vkspec.html#dynamic-state-lifetime">invalidated</a> in the current command buffer prior to this drawing command</li>
     * <li>If the <a href="https://registry.khronos.org/vulkan/specs/latest/html/vkspec.html#features-representativeFragmentTest">{@code representativeFragmentTest}</a> feature is enabled, a shader object is bound to any graphics stage or a graphics pipeline is bound which was created with the {@link EXTExtendedDynamicState3#VK_DYNAMIC_STATE_REPRESENTATIVE_FRAGMENT_TEST_ENABLE_NV DYNAMIC_STATE_REPRESENTATIVE_FRAGMENT_TEST_ENABLE_NV} dynamic state enabled, and the <a href="https://registry.khronos.org/vulkan/specs/latest/html/vkspec.html#dynamic-state-current-value">current value</a> of {@code rasterizerDiscardEnable} is {@link VK10#VK_FALSE FALSE}, then {@link EXTShaderObject#vkCmdSetRepresentativeFragmentTestEnableNV CmdSetRepresentativeFragmentTestEnableNV} <b>must</b> have been called and not subsequently <a href="https://registry.khronos.org/vulkan/specs/latest/html/vkspec.html#dynamic-state-lifetime">invalidated</a> in the current command buffer prior to this drawing command</li>
     * <li>If the <a href="https://registry.khronos.org/vulkan/specs/latest/html/vkspec.html#features-coverageReductionMode">{@code coverageReductionMode}</a> feature is enabled, a shader object is bound to any graphics stage or a graphics pipeline is bound which was created with the {@link EXTExtendedDynamicState3#VK_DYNAMIC_STATE_COVERAGE_REDUCTION_MODE_NV DYNAMIC_STATE_COVERAGE_REDUCTION_MODE_NV} dynamic state enabled, and the <a href="https://registry.khronos.org/vulkan/specs/latest/html/vkspec.html#dynamic-state-current-value">current value</a> of {@code rasterizerDiscardEnable} is {@link VK10#VK_FALSE FALSE}, then {@link EXTShaderObject#vkCmdSetCoverageReductionModeNV CmdSetCoverageReductionModeNV} <b>must</b> have been called and not subsequently <a href="https://registry.khronos.org/vulkan/specs/latest/html/vkspec.html#dynamic-state-lifetime">invalidated</a> in the current command buffer prior to this drawing command</li>
     * <li>If the bound graphics pipeline state was created with the {@link EXTExtendedDynamicState3#VK_DYNAMIC_STATE_COLOR_BLEND_ENABLE_EXT DYNAMIC_STATE_COLOR_BLEND_ENABLE_EXT} state enabled and the last call to {@link EXTShaderObject#vkCmdSetColorBlendEnableEXT CmdSetColorBlendEnableEXT} set {@code pColorBlendEnables} for any attachment to {@link VK10#VK_TRUE TRUE}, then for those attachments in the subpass the corresponding image view’s <a href="https://registry.khronos.org/vulkan/specs/latest/html/vkspec.html#resources-image-view-format-features">format features</a> <b>must</b> contain {@link VK10#VK_FORMAT_FEATURE_COLOR_ATTACHMENT_BLEND_BIT FORMAT_FEATURE_COLOR_ATTACHMENT_BLEND_BIT}</li>
     * <li>If the bound graphics pipeline state was created with the {@link EXTExtendedDynamicState3#VK_DYNAMIC_STATE_RASTERIZATION_SAMPLES_EXT DYNAMIC_STATE_RASTERIZATION_SAMPLES_EXT} state enabled, and the current subpass does not use any color and/or depth/stencil attachments, then the {@code rasterizationSamples} in the last call to {@link EXTShaderObject#vkCmdSetRasterizationSamplesEXT CmdSetRasterizationSamplesEXT} <b>must</b> follow the rules for a <a href="https://registry.khronos.org/vulkan/specs/latest/html/vkspec.html#renderpass-noattachments">zero-attachment subpass</a></li>
     * <li>If the bound graphics pipeline state was created with the {@link EXTExtendedDynamicState3#VK_DYNAMIC_STATE_SAMPLE_MASK_EXT DYNAMIC_STATE_SAMPLE_MASK_EXT} state enabled and the {@link EXTExtendedDynamicState3#VK_DYNAMIC_STATE_RASTERIZATION_SAMPLES_EXT DYNAMIC_STATE_RASTERIZATION_SAMPLES_EXT} state disabled, then the {@code samples} parameter in the last call to {@link EXTShaderObject#vkCmdSetSampleMaskEXT CmdSetSampleMaskEXT} <b>must</b> be greater or equal to the {@link VkPipelineMultisampleStateCreateInfo}{@code ::rasterizationSamples} parameter used to create the bound graphics pipeline</li>
     * <li>If the bound graphics pipeline state was created with the {@link EXTExtendedDynamicState3#VK_DYNAMIC_STATE_SAMPLE_MASK_EXT DYNAMIC_STATE_SAMPLE_MASK_EXT} state and {@link EXTExtendedDynamicState3#VK_DYNAMIC_STATE_RASTERIZATION_SAMPLES_EXT DYNAMIC_STATE_RASTERIZATION_SAMPLES_EXT} states enabled, then the {@code samples} parameter in the last call to {@link EXTShaderObject#vkCmdSetSampleMaskEXT CmdSetSampleMaskEXT} <b>must</b> be greater or equal to the {@code rasterizationSamples} parameter in the last call to {@link EXTShaderObject#vkCmdSetRasterizationSamplesEXT CmdSetRasterizationSamplesEXT}</li>
     * <li>If the bound graphics pipeline state was created with the {@link EXTExtendedDynamicState3#VK_DYNAMIC_STATE_RASTERIZATION_SAMPLES_EXT DYNAMIC_STATE_RASTERIZATION_SAMPLES_EXT} state enabled, and the <a href="https://registry.khronos.org/vulkan/specs/latest/html/vkspec.html#features-multisampledRenderToSingleSampled">{@code multisampledRenderToSingleSampled}</a> feature is not enabled, and neither the {@link AMDMixedAttachmentSamples VK_AMD_mixed_attachment_samples} nor the {@link NVFramebufferMixedSamples VK_NV_framebuffer_mixed_samples} extensions are enabled, then the {@code rasterizationSamples} in the last call to {@link EXTShaderObject#vkCmdSetRasterizationSamplesEXT CmdSetRasterizationSamplesEXT} <b>must</b> be the same as the current subpass color and/or depth/stencil attachments</li>
     * <li>If the bound graphics pipeline state was created with the {@link EXTExtendedDynamicState3#VK_DYNAMIC_STATE_RASTERIZATION_SAMPLES_EXT DYNAMIC_STATE_RASTERIZATION_SAMPLES_EXT} state enabled, or a shader object is bound to any graphics stage, and the current render pass instance includes a {@link VkMultisampledRenderToSingleSampledInfoEXT} structure with {@code multisampledRenderToSingleSampledEnable} equal to {@link VK10#VK_TRUE TRUE}, then the {@code rasterizationSamples} in the last call to {@link EXTShaderObject#vkCmdSetRasterizationSamplesEXT CmdSetRasterizationSamplesEXT} <b>must</b> be the same as the {@code rasterizationSamples} member of that structure</li>
     * <li>If the bound graphics pipeline state was created with the {@link EXTExtendedDynamicState3#VK_DYNAMIC_STATE_COLOR_BLEND_ENABLE_EXT DYNAMIC_STATE_COLOR_BLEND_ENABLE_EXT} dynamic state enabled then {@link EXTShaderObject#vkCmdSetColorBlendEnableEXT CmdSetColorBlendEnableEXT} <b>must</b> have been called in the current command buffer prior to this drawing command, and the attachments specified by the {@code firstAttachment} and {@code attachmentCount} parameters of {@code vkCmdSetColorBlendEnableEXT} calls <b>must</b> specify an enable for all active color attachments in the current subpass</li>
     * <li>If a shader object is bound to the {@link VK10#VK_SHADER_STAGE_FRAGMENT_BIT SHADER_STAGE_FRAGMENT_BIT} stage, and the most recent call to {@link VK13#vkCmdSetRasterizerDiscardEnable CmdSetRasterizerDiscardEnable} in the current command buffer set {@code rasterizerDiscardEnable} to {@link VK10#VK_FALSE FALSE}, then {@link EXTShaderObject#vkCmdSetColorBlendEnableEXT CmdSetColorBlendEnableEXT} <b>must</b> have been called in the current command buffer prior to this drawing command, and the attachments specified by the {@code firstAttachment} and {@code attachmentCount} parameters of {@code vkCmdSetColorBlendEnableEXT} calls <b>must</b> specify an enable for all active color attachments in the current subpass</li>
     * <li>If the bound graphics pipeline state was created with the {@link EXTExtendedDynamicState3#VK_DYNAMIC_STATE_COLOR_BLEND_EQUATION_EXT DYNAMIC_STATE_COLOR_BLEND_EQUATION_EXT} dynamic state enabled then {@link EXTShaderObject#vkCmdSetColorBlendEquationEXT CmdSetColorBlendEquationEXT} <b>must</b> have been called in the current command buffer prior to this drawing command, and the attachments specified by the {@code firstAttachment} and {@code attachmentCount} parameters of {@code vkCmdSetColorBlendEquationEXT} calls <b>must</b> specify the blend equations for all active color attachments in the current subpass where blending is enabled</li>
     * <li>If a shader object is bound to the {@link VK10#VK_SHADER_STAGE_FRAGMENT_BIT SHADER_STAGE_FRAGMENT_BIT} stage, and both the most recent call to {@link VK13#vkCmdSetRasterizerDiscardEnable CmdSetRasterizerDiscardEnable} in the current command buffer set {@code rasterizerDiscardEnable} to {@link VK10#VK_FALSE FALSE} and there are color attachments bound, then {@link EXTShaderObject#vkCmdSetColorBlendEquationEXT CmdSetColorBlendEquationEXT} <b>must</b> have been called in the current command buffer prior to this drawing command, and the attachments specified by the {@code firstAttachment} and {@code attachmentCount} parameters of {@code vkCmdSetColorBlendEquationEXT} calls <b>must</b> specify the blend equations for all active color attachments in the current subpass where blending is enabled</li>
     * <li>If the bound graphics pipeline state was created with the {@link EXTExtendedDynamicState3#VK_DYNAMIC_STATE_COLOR_WRITE_MASK_EXT DYNAMIC_STATE_COLOR_WRITE_MASK_EXT} dynamic state enabled then {@link EXTShaderObject#vkCmdSetColorWriteMaskEXT CmdSetColorWriteMaskEXT} <b>must</b> have been called in the current command buffer prior to this drawing command, and the attachments specified by the {@code firstAttachment} and {@code attachmentCount} parameters of {@code vkCmdSetColorWriteMaskEXT} calls <b>must</b> specify the color write mask for all active color attachments in the current subpass</li>
     * <li>If a shader object is bound to the {@link VK10#VK_SHADER_STAGE_FRAGMENT_BIT SHADER_STAGE_FRAGMENT_BIT} stage, and the most recent call to {@link VK13#vkCmdSetRasterizerDiscardEnable CmdSetRasterizerDiscardEnable} in the current command buffer set {@code rasterizerDiscardEnable} to {@link VK10#VK_FALSE FALSE}, then {@link EXTShaderObject#vkCmdSetColorWriteMaskEXT CmdSetColorWriteMaskEXT} <b>must</b> have been called in the current command buffer prior to this drawing command, and the attachments specified by the {@code firstAttachment} and {@code attachmentCount} parameters of {@code vkCmdSetColorWriteMaskEXT} calls <b>must</b> specify the color write mask for all active color attachments in the current subpass</li>
     * <li>If the bound graphics pipeline state was created with the {@link EXTExtendedDynamicState3#VK_DYNAMIC_STATE_COLOR_BLEND_ADVANCED_EXT DYNAMIC_STATE_COLOR_BLEND_ADVANCED_EXT} dynamic state enabled then {@link EXTShaderObject#vkCmdSetColorBlendAdvancedEXT CmdSetColorBlendAdvancedEXT} <b>must</b> have been called in the current command buffer prior to this drawing command, and the attachments specified by the {@code firstAttachment} and {@code attachmentCount} parameters of {@code vkCmdSetColorBlendAdvancedEXT} calls <b>must</b> specify the advanced blend equations for all active color attachments in the current subpass where blending is enabled</li>
     * <li>If the bound graphics pipeline state was created with the {@link EXTExtendedDynamicState3#VK_DYNAMIC_STATE_COLOR_BLEND_ADVANCED_EXT DYNAMIC_STATE_COLOR_BLEND_ADVANCED_EXT} and {@link EXTExtendedDynamicState3#VK_DYNAMIC_STATE_COLOR_BLEND_ENABLE_EXT DYNAMIC_STATE_COLOR_BLEND_ENABLE_EXT} dynamic states enabled and the last calls to {@link EXTShaderObject#vkCmdSetColorBlendEnableEXT CmdSetColorBlendEnableEXT} and {@link EXTShaderObject#vkCmdSetColorBlendAdvancedEXT CmdSetColorBlendAdvancedEXT} have enabled advanced blending, then the number of active color attachments in the current subpass <b>must</b> not exceed <a href="https://registry.khronos.org/vulkan/specs/latest/html/vkspec.html#limits-advancedBlendMaxColorAttachments">{@code advancedBlendMaxColorAttachments}</a></li>
     * <li>If the <a href="https://registry.khronos.org/vulkan/specs/latest/html/vkspec.html#features-primitivesGeneratedQueryWithNonZeroStreams">{@code primitivesGeneratedQueryWithNonZeroStreams}</a> feature is not enabled and the {@link EXTPrimitivesGeneratedQuery#VK_QUERY_TYPE_PRIMITIVES_GENERATED_EXT QUERY_TYPE_PRIMITIVES_GENERATED_EXT} query is active, and the bound graphics pipeline was created with {@link EXTExtendedDynamicState3#VK_DYNAMIC_STATE_RASTERIZATION_STREAM_EXT DYNAMIC_STATE_RASTERIZATION_STREAM_EXT} state enabled, the last call to {@link EXTShaderObject#vkCmdSetRasterizationStreamEXT CmdSetRasterizationStreamEXT} <b>must</b> have set the {@code rasterizationStream} to zero</li>
     * <li>If the bound graphics pipeline state was created with the {@link EXTSampleLocations#VK_DYNAMIC_STATE_SAMPLE_LOCATIONS_EXT DYNAMIC_STATE_SAMPLE_LOCATIONS_EXT} state enabled and the {@link EXTExtendedDynamicState3#VK_DYNAMIC_STATE_RASTERIZATION_SAMPLES_EXT DYNAMIC_STATE_RASTERIZATION_SAMPLES_EXT} state disabled, and the <a href="https://registry.khronos.org/vulkan/specs/latest/html/vkspec.html#dynamic-state-current-value">current value</a> of {@code sampleLocationsEnable} is {@link VK10#VK_TRUE TRUE}, then the {@code sampleLocationsPerPixel} member of {@code pSampleLocationsInfo} in the last call to {@link EXTSampleLocations#vkCmdSetSampleLocationsEXT CmdSetSampleLocationsEXT} <b>must</b> equal the {@code rasterizationSamples} member of the {@link VkPipelineMultisampleStateCreateInfo} structure the bound graphics pipeline has been created with</li>
     * <li>If the bound graphics pipeline state was created with the {@link EXTSampleLocations#VK_DYNAMIC_STATE_SAMPLE_LOCATIONS_EXT DYNAMIC_STATE_SAMPLE_LOCATIONS_EXT} state enabled and the {@link EXTExtendedDynamicState3#VK_DYNAMIC_STATE_RASTERIZATION_SAMPLES_EXT DYNAMIC_STATE_RASTERIZATION_SAMPLES_EXT} state enabled, and the <a href="https://registry.khronos.org/vulkan/specs/latest/html/vkspec.html#dynamic-state-current-value">current value</a> of {@code sampleLocationsEnable} is {@link VK10#VK_TRUE TRUE}, then the {@code sampleLocationsPerPixel} member of {@code pSampleLocationsInfo} in the last call to {@link EXTSampleLocations#vkCmdSetSampleLocationsEXT CmdSetSampleLocationsEXT} <b>must</b> equal the {@code rasterizationSamples} parameter of the last call to {@link EXTShaderObject#vkCmdSetRasterizationSamplesEXT CmdSetRasterizationSamplesEXT}</li>
     * <li>If a shader object is bound to the {@link VK10#VK_SHADER_STAGE_FRAGMENT_BIT SHADER_STAGE_FRAGMENT_BIT} stage, or the bound graphics pipeline was created with the {@link EXTExtendedDynamicState3#VK_DYNAMIC_STATE_SAMPLE_LOCATIONS_ENABLE_EXT DYNAMIC_STATE_SAMPLE_LOCATIONS_ENABLE_EXT} state enabled, and {@code sampleLocationsEnable} was {@link VK10#VK_TRUE TRUE} in the last call to {@link EXTShaderObject#vkCmdSetSampleLocationsEnableEXT CmdSetSampleLocationsEnableEXT}, and the current subpass has a depth/stencil attachment, then that attachment <b>must</b> have been created with the {@link EXTSampleLocations#VK_IMAGE_CREATE_SAMPLE_LOCATIONS_COMPATIBLE_DEPTH_BIT_EXT IMAGE_CREATE_SAMPLE_LOCATIONS_COMPATIBLE_DEPTH_BIT_EXT} bit set</li>
     * <li>If a shader object is bound to the {@link VK10#VK_SHADER_STAGE_FRAGMENT_BIT SHADER_STAGE_FRAGMENT_BIT} stage, or the bound graphics pipeline state was created with the {@link EXTSampleLocations#VK_DYNAMIC_STATE_SAMPLE_LOCATIONS_EXT DYNAMIC_STATE_SAMPLE_LOCATIONS_EXT} state enabled and the {@link EXTExtendedDynamicState3#VK_DYNAMIC_STATE_SAMPLE_LOCATIONS_ENABLE_EXT DYNAMIC_STATE_SAMPLE_LOCATIONS_ENABLE_EXT} state enabled, and if {@code sampleLocationsEnable} was {@link VK10#VK_TRUE TRUE} in the last call to {@link EXTShaderObject#vkCmdSetSampleLocationsEnableEXT CmdSetSampleLocationsEnableEXT}, then the {@code sampleLocationsInfo.sampleLocationGridSize.width} in the last call to {@link EXTSampleLocations#vkCmdSetSampleLocationsEXT CmdSetSampleLocationsEXT} <b>must</b> evenly divide {@link VkMultisamplePropertiesEXT}{@code ::sampleLocationGridSize.width} as returned by {@link EXTSampleLocations#vkGetPhysicalDeviceMultisamplePropertiesEXT GetPhysicalDeviceMultisamplePropertiesEXT} with a {@code samples} parameter equaling {@code rasterizationSamples}</li>
     * <li>If a shader object is bound to the {@link VK10#VK_SHADER_STAGE_FRAGMENT_BIT SHADER_STAGE_FRAGMENT_BIT} stage, or the bound graphics pipeline state was created with the {@link EXTSampleLocations#VK_DYNAMIC_STATE_SAMPLE_LOCATIONS_EXT DYNAMIC_STATE_SAMPLE_LOCATIONS_EXT} state enabled and the {@link EXTExtendedDynamicState3#VK_DYNAMIC_STATE_SAMPLE_LOCATIONS_ENABLE_EXT DYNAMIC_STATE_SAMPLE_LOCATIONS_ENABLE_EXT} state enabled, and if {@code sampleLocationsEnable} was {@link VK10#VK_TRUE TRUE} in the last call to {@link EXTShaderObject#vkCmdSetSampleLocationsEnableEXT CmdSetSampleLocationsEnableEXT}, then the {@code sampleLocationsInfo.sampleLocationGridSize.height} in the last call to {@link EXTSampleLocations#vkCmdSetSampleLocationsEXT CmdSetSampleLocationsEXT} <b>must</b> evenly divide {@link VkMultisamplePropertiesEXT}{@code ::sampleLocationGridSize.height} as returned by {@link EXTSampleLocations#vkGetPhysicalDeviceMultisamplePropertiesEXT GetPhysicalDeviceMultisamplePropertiesEXT} with a {@code samples} parameter equaling {@code rasterizationSamples}</li>
     * <li>If a shader object is bound to the {@link VK10#VK_SHADER_STAGE_FRAGMENT_BIT SHADER_STAGE_FRAGMENT_BIT} stage, or the bound graphics pipeline state was created with the {@link EXTExtendedDynamicState3#VK_DYNAMIC_STATE_SAMPLE_LOCATIONS_ENABLE_EXT DYNAMIC_STATE_SAMPLE_LOCATIONS_ENABLE_EXT} state enabled, and if {@code sampleLocationsEnable} was {@link VK10#VK_TRUE TRUE} in the last call to {@link EXTShaderObject#vkCmdSetSampleLocationsEnableEXT CmdSetSampleLocationsEnableEXT}, the fragment shader code <b>must</b> not statically use the extended instruction {@code InterpolateAtSample}</li>
     * <li>If the bound graphics pipeline state was created with the {@link EXTSampleLocations#VK_DYNAMIC_STATE_SAMPLE_LOCATIONS_EXT DYNAMIC_STATE_SAMPLE_LOCATIONS_EXT} state disabled and the {@link EXTExtendedDynamicState3#VK_DYNAMIC_STATE_RASTERIZATION_SAMPLES_EXT DYNAMIC_STATE_RASTERIZATION_SAMPLES_EXT} state enabled, and the <a href="https://registry.khronos.org/vulkan/specs/latest/html/vkspec.html#dynamic-state-current-value">current value</a> of {@code sampleLocationsEnable} is {@link VK10#VK_TRUE TRUE}, then {@code sampleLocationsInfo.sampleLocationGridSize.width} <b>must</b> evenly divide {@link VkMultisamplePropertiesEXT}{@code ::sampleLocationGridSize.width} as returned by {@link EXTSampleLocations#vkGetPhysicalDeviceMultisamplePropertiesEXT GetPhysicalDeviceMultisamplePropertiesEXT} with a {@code samples} parameter equaling the value of {@code rasterizationSamples} in the last call to {@link EXTShaderObject#vkCmdSetRasterizationSamplesEXT CmdSetRasterizationSamplesEXT}</li>
     * <li>If the bound graphics pipeline state was created with the {@link EXTSampleLocations#VK_DYNAMIC_STATE_SAMPLE_LOCATIONS_EXT DYNAMIC_STATE_SAMPLE_LOCATIONS_EXT} state disabled and the {@link EXTExtendedDynamicState3#VK_DYNAMIC_STATE_RASTERIZATION_SAMPLES_EXT DYNAMIC_STATE_RASTERIZATION_SAMPLES_EXT} state enabled, and the <a href="https://registry.khronos.org/vulkan/specs/latest/html/vkspec.html#dynamic-state-current-value">current value</a> of {@code sampleLocationsEnable} is {@link VK10#VK_TRUE TRUE}, then {@code sampleLocationsInfo.sampleLocationGridSize.height} <b>must</b> evenly divide {@link VkMultisamplePropertiesEXT}{@code ::sampleLocationGridSize.height} as returned by {@link EXTSampleLocations#vkGetPhysicalDeviceMultisamplePropertiesEXT GetPhysicalDeviceMultisamplePropertiesEXT} with a {@code samples} parameter equaling the value of {@code rasterizationSamples} in the last call to {@link EXTShaderObject#vkCmdSetRasterizationSamplesEXT CmdSetRasterizationSamplesEXT}</li>
     * <li>If the bound graphics pipeline state was created with the {@link EXTSampleLocations#VK_DYNAMIC_STATE_SAMPLE_LOCATIONS_EXT DYNAMIC_STATE_SAMPLE_LOCATIONS_EXT} state disabled and the {@link EXTExtendedDynamicState3#VK_DYNAMIC_STATE_RASTERIZATION_SAMPLES_EXT DYNAMIC_STATE_RASTERIZATION_SAMPLES_EXT} state enabled, and the <a href="https://registry.khronos.org/vulkan/specs/latest/html/vkspec.html#dynamic-state-current-value">current value</a> of {@code sampleLocationsEnable} is {@link VK10#VK_TRUE TRUE}, then {@code sampleLocationsInfo.sampleLocationsPerPixel} <b>must</b> equal {@code rasterizationSamples} in the last call to {@link EXTShaderObject#vkCmdSetRasterizationSamplesEXT CmdSetRasterizationSamplesEXT}</li>
     * <li>If a shader object is bound to any graphics stage or the bound graphics pipeline state was created with the {@link EXTExtendedDynamicState3#VK_DYNAMIC_STATE_COVERAGE_MODULATION_TABLE_ENABLE_NV DYNAMIC_STATE_COVERAGE_MODULATION_TABLE_ENABLE_NV} state enabled, and the last call to {@link EXTShaderObject#vkCmdSetCoverageModulationTableEnableNV CmdSetCoverageModulationTableEnableNV} set {@code coverageModulationTableEnable} to {@link VK10#VK_TRUE TRUE}, then the {@code coverageModulationTableCount} parameter in the last call to {@link EXTShaderObject#vkCmdSetCoverageModulationTableNV CmdSetCoverageModulationTableNV} <b>must</b> equal the current {@code rasterizationSamples} divided by the number of color samples in the current subpass</li>
     * <li>If the {@link NVFramebufferMixedSamples VK_NV_framebuffer_mixed_samples} extension is enabled, and if current subpass has a depth/stencil attachment and depth test, stencil test, or depth bounds test are enabled in the bound pipeline, then the current {@code rasterizationSamples} <b>must</b> be the same as the sample count of the depth/stencil attachment</li>
     * <li>If the bound graphics pipeline state was created with the {@link EXTExtendedDynamicState3#VK_DYNAMIC_STATE_COVERAGE_TO_COLOR_ENABLE_NV DYNAMIC_STATE_COVERAGE_TO_COLOR_ENABLE_NV} state enabled and the last call to {@link EXTShaderObject#vkCmdSetCoverageToColorEnableNV CmdSetCoverageToColorEnableNV} set the {@code coverageToColorEnable} to {@link VK10#VK_TRUE TRUE}, then the current subpass <b>must</b> have a color attachment at the location selected by the last call to {@link EXTShaderObject#vkCmdSetCoverageToColorLocationNV CmdSetCoverageToColorLocationNV} {@code coverageToColorLocation}, with a {@code VkFormat} of {@link VK10#VK_FORMAT_R8_UINT FORMAT_R8_UINT}, {@link VK10#VK_FORMAT_R8_SINT FORMAT_R8_SINT}, {@link VK10#VK_FORMAT_R16_UINT FORMAT_R16_UINT}, {@link VK10#VK_FORMAT_R16_SINT FORMAT_R16_SINT}, {@link VK10#VK_FORMAT_R32_UINT FORMAT_R32_UINT}, or {@link VK10#VK_FORMAT_R32_SINT FORMAT_R32_SINT}</li>
     * <li>If the {@link NVFragmentCoverageToColor VK_NV_fragment_coverage_to_color} extension is enabled, and a shader object is bound to the {@link VK10#VK_SHADER_STAGE_FRAGMENT_BIT SHADER_STAGE_FRAGMENT_BIT} stage, and the most recent call to {@link VK13#vkCmdSetRasterizerDiscardEnable CmdSetRasterizerDiscardEnable} in the current command buffer set {@code rasterizerDiscardEnable} to {@link VK10#VK_FALSE FALSE}, and the last call to {@link EXTShaderObject#vkCmdSetCoverageToColorEnableNV CmdSetCoverageToColorEnableNV} set the {@code coverageToColorEnable} to {@link VK10#VK_TRUE TRUE}, then the current subpass <b>must</b> have a color attachment at the location selected by the last call to {@link EXTShaderObject#vkCmdSetCoverageToColorLocationNV CmdSetCoverageToColorLocationNV} {@code coverageToColorLocation}, with a {@code VkFormat} of {@link VK10#VK_FORMAT_R8_UINT FORMAT_R8_UINT}, {@link VK10#VK_FORMAT_R8_SINT FORMAT_R8_SINT}, {@link VK10#VK_FORMAT_R16_UINT FORMAT_R16_UINT}, {@link VK10#VK_FORMAT_R16_SINT FORMAT_R16_SINT}, {@link VK10#VK_FORMAT_R32_UINT FORMAT_R32_UINT}, or {@link VK10#VK_FORMAT_R32_SINT FORMAT_R32_SINT}</li>
     * <li>If this {@link NVCoverageReductionMode VK_NV_coverage_reduction_mode} extension is enabled, the bound graphics pipeline state was created with the {@link EXTExtendedDynamicState3#VK_DYNAMIC_STATE_COVERAGE_TO_COLOR_ENABLE_NV DYNAMIC_STATE_COVERAGE_TO_COLOR_ENABLE_NV} and {@link EXTExtendedDynamicState3#VK_DYNAMIC_STATE_RASTERIZATION_SAMPLES_EXT DYNAMIC_STATE_RASTERIZATION_SAMPLES_EXT} states enabled, the current coverage reduction mode {@code coverageReductionMode}, then the current {@code rasterizationSamples}, and the sample counts for the color and depth/stencil attachments (if the subpass has them) <b>must</b> be a valid combination returned by {@link NVCoverageReductionMode#vkGetPhysicalDeviceSupportedFramebufferMixedSamplesCombinationsNV GetPhysicalDeviceSupportedFramebufferMixedSamplesCombinationsNV}</li>
     * <li>If the bound graphics pipeline state was created with the {@link VK13#VK_DYNAMIC_STATE_VIEWPORT_WITH_COUNT DYNAMIC_STATE_VIEWPORT_WITH_COUNT} dynamic state enabled, but not the {@link EXTExtendedDynamicState3#VK_DYNAMIC_STATE_VIEWPORT_SWIZZLE_NV DYNAMIC_STATE_VIEWPORT_SWIZZLE_NV} dynamic state enabled, then the bound graphics pipeline <b>must</b> have been created with {@link VkPipelineViewportSwizzleStateCreateInfoNV}{@code ::viewportCount} greater or equal to the {@code viewportCount} parameter in the last call to {@link VK13#vkCmdSetViewportWithCount CmdSetViewportWithCount}</li>
     * <li>If the bound graphics pipeline state was created with the {@link VK13#VK_DYNAMIC_STATE_VIEWPORT_WITH_COUNT DYNAMIC_STATE_VIEWPORT_WITH_COUNT} and {@link EXTExtendedDynamicState3#VK_DYNAMIC_STATE_VIEWPORT_SWIZZLE_NV DYNAMIC_STATE_VIEWPORT_SWIZZLE_NV} dynamic states enabled then the {@code viewportCount} parameter in the last call to {@link EXTShaderObject#vkCmdSetViewportSwizzleNV CmdSetViewportSwizzleNV} <b>must</b> be greater than or equal to the {@code viewportCount} parameter in the last call to {@link VK13#vkCmdSetViewportWithCount CmdSetViewportWithCount}</li>
     * <li>If the {@link NVViewportSwizzle VK_NV_viewport_swizzle} extension is enabled, and a shader object is bound to any graphics stage, then the {@code viewportCount} parameter in the last call to {@link EXTShaderObject#vkCmdSetViewportSwizzleNV CmdSetViewportSwizzleNV} <b>must</b> be greater than or equal to the {@code viewportCount} parameter in the last call to {@link VK13#vkCmdSetViewportWithCount CmdSetViewportWithCount}</li>
     * <li>If the {@link NVFramebufferMixedSamples VK_NV_framebuffer_mixed_samples} extension is enabled, and if the current subpass has any color attachments and {@code rasterizationSamples} of the last call to {@link EXTShaderObject#vkCmdSetRasterizationSamplesEXT CmdSetRasterizationSamplesEXT} is greater than the number of color samples, then the pipeline {@code sampleShadingEnable} <b>must</b> be {@link VK10#VK_FALSE FALSE}</li>
     * <li>If the bound graphics pipeline state was created with the {@link EXTExtendedDynamicState3#VK_DYNAMIC_STATE_LINE_STIPPLE_ENABLE_EXT DYNAMIC_STATE_LINE_STIPPLE_ENABLE_EXT} or {@link EXTExtendedDynamicState3#VK_DYNAMIC_STATE_LINE_RASTERIZATION_MODE_EXT DYNAMIC_STATE_LINE_RASTERIZATION_MODE_EXT} dynamic states enabled, and if the current {@code stippledLineEnable} state is {@link VK10#VK_TRUE TRUE} and the current {@code lineRasterizationMode} state is {@link VK14#VK_LINE_RASTERIZATION_MODE_RECTANGULAR LINE_RASTERIZATION_MODE_RECTANGULAR}, then the <a href="https://registry.khronos.org/vulkan/specs/latest/html/vkspec.html#features-stippledRectangularLines">{@code stippledRectangularLines}</a> feature <b>must</b> be enabled</li>
     * <li>If the bound graphics pipeline state was created with the {@link EXTExtendedDynamicState3#VK_DYNAMIC_STATE_LINE_STIPPLE_ENABLE_EXT DYNAMIC_STATE_LINE_STIPPLE_ENABLE_EXT} or {@link EXTExtendedDynamicState3#VK_DYNAMIC_STATE_LINE_RASTERIZATION_MODE_EXT DYNAMIC_STATE_LINE_RASTERIZATION_MODE_EXT} dynamic states enabled, and if the current {@code stippledLineEnable} state is {@link VK10#VK_TRUE TRUE} and the current {@code lineRasterizationMode} state is {@link VK14#VK_LINE_RASTERIZATION_MODE_BRESENHAM LINE_RASTERIZATION_MODE_BRESENHAM}, then the <a href="https://registry.khronos.org/vulkan/specs/latest/html/vkspec.html#features-stippledBresenhamLines">{@code stippledBresenhamLines}</a> feature <b>must</b> be enabled</li>
     * <li>If the bound graphics pipeline state was created with the {@link EXTExtendedDynamicState3#VK_DYNAMIC_STATE_LINE_STIPPLE_ENABLE_EXT DYNAMIC_STATE_LINE_STIPPLE_ENABLE_EXT} or {@link EXTExtendedDynamicState3#VK_DYNAMIC_STATE_LINE_RASTERIZATION_MODE_EXT DYNAMIC_STATE_LINE_RASTERIZATION_MODE_EXT} dynamic states enabled, and if the current {@code stippledLineEnable} state is {@link VK10#VK_TRUE TRUE} and the current {@code lineRasterizationMode} state is {@link VK14#VK_LINE_RASTERIZATION_MODE_RECTANGULAR_SMOOTH LINE_RASTERIZATION_MODE_RECTANGULAR_SMOOTH}, then the <a href="https://registry.khronos.org/vulkan/specs/latest/html/vkspec.html#features-stippledSmoothLines">{@code stippledSmoothLines}</a> feature <b>must</b> be enabled</li>
     * <li>If the bound graphics pipeline state was created with the {@link EXTExtendedDynamicState3#VK_DYNAMIC_STATE_LINE_STIPPLE_ENABLE_EXT DYNAMIC_STATE_LINE_STIPPLE_ENABLE_EXT} or {@link EXTExtendedDynamicState3#VK_DYNAMIC_STATE_LINE_RASTERIZATION_MODE_EXT DYNAMIC_STATE_LINE_RASTERIZATION_MODE_EXT} dynamic states enabled, and if the current {@code stippledLineEnable} state is {@link VK10#VK_TRUE TRUE} and the current {@code lineRasterizationMode} state is {@link VK14#VK_LINE_RASTERIZATION_MODE_DEFAULT LINE_RASTERIZATION_MODE_DEFAULT}, then the <a href="https://registry.khronos.org/vulkan/specs/latest/html/vkspec.html#features-stippledRectangularLines">{@code stippledRectangularLines}</a> feature <b>must</b> be enabled and {@link VkPhysicalDeviceLimits}{@code ::strictLines} <b>must</b> be {@link VK10#VK_TRUE TRUE}</li>
     * <li>If the bound graphics pipeline state was created with the {@link EXTExtendedDynamicState3#VK_DYNAMIC_STATE_CONSERVATIVE_RASTERIZATION_MODE_EXT DYNAMIC_STATE_CONSERVATIVE_RASTERIZATION_MODE_EXT} dynamic state enabled, <a href="https://registry.khronos.org/vulkan/specs/latest/html/vkspec.html#limits-conservativePointAndLineRasterization">{@code conservativePointAndLineRasterization}</a> is not supported, and the effective primitive topology output by the last pre-rasterization shader stage is a line or point, then the {@code conservativeRasterizationMode} set by the last call to {@link EXTShaderObject#vkCmdSetConservativeRasterizationModeEXT CmdSetConservativeRasterizationModeEXT} <b>must</b> be {@link EXTConservativeRasterization#VK_CONSERVATIVE_RASTERIZATION_MODE_DISABLED_EXT CONSERVATIVE_RASTERIZATION_MODE_DISABLED_EXT}</li>
     * <li>If the bound pipeline was created with the {@link VkPipelineShaderStageCreateInfo}{@code ::stage} member of an element of {@link VkGraphicsPipelineCreateInfo}{@code ::pStages} set to {@link VK10#VK_SHADER_STAGE_VERTEX_BIT SHADER_STAGE_VERTEX_BIT}, {@link VK10#VK_SHADER_STAGE_TESSELLATION_CONTROL_BIT SHADER_STAGE_TESSELLATION_CONTROL_BIT}, {@link VK10#VK_SHADER_STAGE_TESSELLATION_EVALUATION_BIT SHADER_STAGE_TESSELLATION_EVALUATION_BIT} or {@link VK10#VK_SHADER_STAGE_GEOMETRY_BIT SHADER_STAGE_GEOMETRY_BIT}, then <a href="https://registry.khronos.org/vulkan/specs/latest/html/vkspec.html#queries-mesh-shader">Mesh Shader Queries</a> <b>must</b> not be active</li>
     * <li>If a shader object is bound to the {@link VK10#VK_SHADER_STAGE_FRAGMENT_BIT SHADER_STAGE_FRAGMENT_BIT} stage or a graphics pipeline is bound which was created with the {@link EXTAttachmentFeedbackLoopDynamicState#VK_DYNAMIC_STATE_ATTACHMENT_FEEDBACK_LOOP_ENABLE_EXT DYNAMIC_STATE_ATTACHMENT_FEEDBACK_LOOP_ENABLE_EXT} dynamic state enabled, and the <a href="https://registry.khronos.org/vulkan/specs/latest/html/vkspec.html#dynamic-state-current-value">current value</a> of {@code rasterizerDiscardEnable} is {@link VK10#VK_FALSE FALSE}, then {@link EXTAttachmentFeedbackLoopDynamicState#vkCmdSetAttachmentFeedbackLoopEnableEXT CmdSetAttachmentFeedbackLoopEnableEXT} <b>must</b> have been called and not subsequently <a href="https://registry.khronos.org/vulkan/specs/latest/html/vkspec.html#dynamic-state-lifetime">invalidated</a> in the current command buffer prior to this drawing command</li>
     * <li>If dynamic state was inherited from {@link VkCommandBufferInheritanceViewportScissorInfoNV}, it <b>must</b> be set in the current command buffer prior to this drawing command</li>
     * <li>If there is no bound graphics pipeline, {@code vkCmdBindShadersEXT} <b>must</b> have been called in the current command buffer with {@code pStages} with an element of {@link VK10#VK_SHADER_STAGE_FRAGMENT_BIT SHADER_STAGE_FRAGMENT_BIT}</li>
     * <li>If there is no bound graphics pipeline, and the <a href="https://registry.khronos.org/vulkan/specs/latest/html/vkspec.html#features-taskShader">{@code taskShader}</a> feature is enabled, {@code vkCmdBindShadersEXT} <b>must</b> have been called in the current command buffer with {@code pStages} with an element of {@link EXTMeshShader#VK_SHADER_STAGE_TASK_BIT_EXT SHADER_STAGE_TASK_BIT_EXT}</li>
     * <li>If there is no bound graphics pipeline, and the <a href="https://registry.khronos.org/vulkan/specs/latest/html/vkspec.html#features-meshShader">{@code meshShader}</a> feature is enabled, {@code vkCmdBindShadersEXT} <b>must</b> have been called in the current command buffer with {@code pStages} with an element of {@link EXTMeshShader#VK_SHADER_STAGE_MESH_BIT_EXT SHADER_STAGE_MESH_BIT_EXT}</li>
     * <li>If there is no bound graphics pipeline, and at least one of the <a href="https://registry.khronos.org/vulkan/specs/latest/html/vkspec.html#features-taskShader">{@code taskShader}</a> and <a href="https://registry.khronos.org/vulkan/specs/latest/html/vkspec.html#features-meshShader">{@code meshShader}</a> features is enabled, one of the {@link VK10#VK_SHADER_STAGE_VERTEX_BIT SHADER_STAGE_VERTEX_BIT} or {@link EXTMeshShader#VK_SHADER_STAGE_MESH_BIT_EXT SHADER_STAGE_MESH_BIT_EXT} stages <b>must</b> have a valid {@code VkShaderEXT} bound, and the other <b>must</b> have no {@code VkShaderEXT} bound</li>
     * <li>If there is no bound graphics pipeline, and both the <a href="https://registry.khronos.org/vulkan/specs/latest/html/vkspec.html#features-taskShader">{@code taskShader}</a> and <a href="https://registry.khronos.org/vulkan/specs/latest/html/vkspec.html#features-meshShader">{@code meshShader}</a> features are enabled, and a valid {@code VkShaderEXT} is bound the to the {@link EXTMeshShader#VK_SHADER_STAGE_MESH_BIT_EXT SHADER_STAGE_MESH_BIT_EXT} stage, and that {@code VkShaderEXT} was created without the {@link EXTShaderObject#VK_SHADER_CREATE_NO_TASK_SHADER_BIT_EXT SHADER_CREATE_NO_TASK_SHADER_BIT_EXT} flag, a valid {@code VkShaderEXT} <b>must</b> be bound to the {@link EXTMeshShader#VK_SHADER_STAGE_TASK_BIT_EXT SHADER_STAGE_TASK_BIT_EXT} stage</li>
     * <li>If there is no bound graphics pipeline, and both the <a href="https://registry.khronos.org/vulkan/specs/latest/html/vkspec.html#features-taskShader">{@code taskShader}</a> and <a href="https://registry.khronos.org/vulkan/specs/latest/html/vkspec.html#features-meshShader">{@code meshShader}</a> features are enabled, and a valid {@code VkShaderEXT} is bound the to the {@link EXTMeshShader#VK_SHADER_STAGE_MESH_BIT_EXT SHADER_STAGE_MESH_BIT_EXT} stage, and that {@code VkShaderEXT} was created with the {@link EXTShaderObject#VK_SHADER_CREATE_NO_TASK_SHADER_BIT_EXT SHADER_CREATE_NO_TASK_SHADER_BIT_EXT} flag, there <b>must</b> be no {@code VkShaderEXT} bound to the {@link EXTMeshShader#VK_SHADER_STAGE_TASK_BIT_EXT SHADER_STAGE_TASK_BIT_EXT} stage</li>
     * <li>If there is no bound graphics pipeline, and a valid {@code VkShaderEXT} is bound to the {@link VK10#VK_SHADER_STAGE_VERTEX_BIT SHADER_STAGE_VERTEX_BIT} stage, there <b>must</b> be no {@code VkShaderEXT} bound to either the {@link EXTMeshShader#VK_SHADER_STAGE_TASK_BIT_EXT SHADER_STAGE_TASK_BIT_EXT} stage or the {@link EXTMeshShader#VK_SHADER_STAGE_MESH_BIT_EXT SHADER_STAGE_MESH_BIT_EXT} stage</li>
     * <li>If any graphics shader is bound which was created with the {@link EXTShaderObject#VK_SHADER_CREATE_LINK_STAGE_BIT_EXT SHADER_CREATE_LINK_STAGE_BIT_EXT} flag, then all shaders created with the {@link EXTShaderObject#VK_SHADER_CREATE_LINK_STAGE_BIT_EXT SHADER_CREATE_LINK_STAGE_BIT_EXT} flag in the same {@link EXTShaderObject#vkCreateShadersEXT CreateShadersEXT} call <b>must</b> also be bound</li>
     * <li>If any graphics shader is bound which was created with the {@link EXTShaderObject#VK_SHADER_CREATE_LINK_STAGE_BIT_EXT SHADER_CREATE_LINK_STAGE_BIT_EXT} flag, any stages in between stages whose shaders which did not create a shader with the {@link EXTShaderObject#VK_SHADER_CREATE_LINK_STAGE_BIT_EXT SHADER_CREATE_LINK_STAGE_BIT_EXT} flag as part of the same {@link EXTShaderObject#vkCreateShadersEXT CreateShadersEXT} call <b>must</b> not have any {@code VkShaderEXT} bound</li>
     * <li>All bound graphics shader objects <b>must</b> have been created with identical or identically defined push constant ranges</li>
     * <li>All bound graphics shader objects <b>must</b> have been created with identical or identically defined arrays of descriptor set layouts</li>
     * <li>If the current render pass instance was begun with {@link VK13#vkCmdBeginRendering CmdBeginRendering} and a {@link VkRenderingInfo}{@code ::colorAttachmentCount} equal to 1, a color attachment with a resolve mode of {@link ANDROIDExternalFormatResolve#VK_RESOLVE_MODE_EXTERNAL_FORMAT_DOWNSAMPLE_ANDROID RESOLVE_MODE_EXTERNAL_FORMAT_DOWNSAMPLE_ANDROID}, and a fragment shader is bound, it <b>must</b> not declare the {@code DepthReplacing} or {@code StencilRefReplacingEXT} execution modes</li>
     * <li>If the bound graphics pipeline state includes a fragment shader stage, was created with {@link VK13#VK_DYNAMIC_STATE_DEPTH_WRITE_ENABLE DYNAMIC_STATE_DEPTH_WRITE_ENABLE} set in {@link VkPipelineDynamicStateCreateInfo}{@code ::pDynamicStates}, and the fragment shader declares the {@code EarlyFragmentTests} execution mode and uses {@code OpDepthAttachmentReadEXT}, the {@code depthWriteEnable} parameter in the last call to {@link VK13#vkCmdSetDepthWriteEnable CmdSetDepthWriteEnable} <b>must</b> be {@link VK10#VK_FALSE FALSE}</li>
     * <li>If the bound graphics pipeline state includes a fragment shader stage, was created with {@link VK10#VK_DYNAMIC_STATE_STENCIL_WRITE_MASK DYNAMIC_STATE_STENCIL_WRITE_MASK} set in {@link VkPipelineDynamicStateCreateInfo}{@code ::pDynamicStates}, and the fragment shader declares the {@code EarlyFragmentTests} execution mode and uses {@code OpStencilAttachmentReadEXT}, the {@code writeMask} parameter in the last call to {@link VK10#vkCmdSetStencilWriteMask CmdSetStencilWriteMask} <b>must</b> be 0</li>
     * <li>If a shader object is bound to any graphics stage or the bound graphics pipeline was created with {@link EXTExtendedDynamicState3#VK_DYNAMIC_STATE_COLOR_WRITE_MASK_EXT DYNAMIC_STATE_COLOR_WRITE_MASK_EXT}, and the format of any color attachment is {@link VK10#VK_FORMAT_E5B9G9R9_UFLOAT_PACK32 FORMAT_E5B9G9R9_UFLOAT_PACK32}, the corresponding element of the {@code pColorWriteMasks} parameter of {@link EXTShaderObject#vkCmdSetColorWriteMaskEXT CmdSetColorWriteMaskEXT} <b>must</b> either include all of {@link VK10#VK_COLOR_COMPONENT_R_BIT COLOR_COMPONENT_R_BIT}, {@link VK10#VK_COLOR_COMPONENT_G_BIT COLOR_COMPONENT_G_BIT}, and {@link VK10#VK_COLOR_COMPONENT_B_BIT COLOR_COMPONENT_B_BIT}, or none of them</li>
     * <li>If <a href="https://registry.khronos.org/vulkan/specs/latest/html/vkspec.html#framebuffer-blending">blending</a> is enabled for any attachment where either the source or destination blend factors for that attachment <a href="https://registry.khronos.org/vulkan/specs/latest/html/vkspec.html#framebuffer-dsb">use the secondary color input</a>, the maximum value of {@code Location} for any output attachment <a href="https://registry.khronos.org/vulkan/specs/latest/html/vkspec.html#shaders-staticuse">statically used</a> in the {@code Fragment} {@code Execution} {@code Model} executed by this command <b>must</b> be less than <a href="https://registry.khronos.org/vulkan/specs/latest/html/vkspec.html#limits-maxFragmentDualSrcAttachments">{@code maxFragmentDualSrcAttachments}</a></li>
     * <li>If the current render pass was begun with {@link VK13#vkCmdBeginRendering CmdBeginRendering}, and there is no shader object bound to any graphics stage, the value of each element of {@link VkRenderingAttachmentLocationInfo}{@code ::pColorAttachmentLocations} set by {@link VK14#vkCmdSetRenderingAttachmentLocations CmdSetRenderingAttachmentLocations} <b>must</b> match the value set for the corresponding element in the bound pipeline</li>
     * <li>If the current render pass was begun with {@link VK13#vkCmdBeginRendering CmdBeginRendering}, and there is no shader object bound to any graphics stage, input attachment index mappings in the bound pipeline <b>must</b> match those set for the current render pass instance via {@link VkRenderingInputAttachmentIndexInfo}</li>
     * <li>If the current render pass was begun with {@link VK13#vkCmdBeginRendering CmdBeginRendering} with the {@link EXTLegacyDithering#VK_RENDERING_ENABLE_LEGACY_DITHERING_BIT_EXT RENDERING_ENABLE_LEGACY_DITHERING_BIT_EXT} flag, the bound graphics pipeline <b>must</b> have been created with {@link KHRMaintenance5#VK_PIPELINE_CREATE_2_ENABLE_LEGACY_DITHERING_BIT_EXT PIPELINE_CREATE_2_ENABLE_LEGACY_DITHERING_BIT_EXT}</li>
     * <li>If the bound graphics pipeline was created with {@link KHRMaintenance5#VK_PIPELINE_CREATE_2_ENABLE_LEGACY_DITHERING_BIT_EXT PIPELINE_CREATE_2_ENABLE_LEGACY_DITHERING_BIT_EXT}, the current render pass <b>must</b> have begun with {@link VK13#vkCmdBeginRendering CmdBeginRendering} with the {@link EXTLegacyDithering#VK_RENDERING_ENABLE_LEGACY_DITHERING_BIT_EXT RENDERING_ENABLE_LEGACY_DITHERING_BIT_EXT} flag</li>
     * </ul>
     * 
     * <ul>
     * <li>All vertex input bindings accessed via vertex input variables declared in the vertex shader entry point’s interface <b>must</b> have either valid or {@link VK10#VK_NULL_HANDLE NULL_HANDLE} buffers bound</li>
     * <li>If the <a href="https://registry.khronos.org/vulkan/specs/latest/html/vkspec.html#features-nullDescriptor">{@code nullDescriptor}</a> feature is not enabled, all vertex input bindings accessed via vertex input variables declared in the vertex shader entry point’s interface <b>must</b> not be {@link VK10#VK_NULL_HANDLE NULL_HANDLE}</li>
     * <li>If the <a href="https://registry.khronos.org/vulkan/specs/latest/html/vkspec.html#features-robustBufferAccess">{@code robustBufferAccess}</a> feature is not enabled, and that pipeline was created without enabling {@link VK14#VK_PIPELINE_ROBUSTNESS_BUFFER_BEHAVIOR_ROBUST_BUFFER_ACCESS PIPELINE_ROBUSTNESS_BUFFER_BEHAVIOR_ROBUST_BUFFER_ACCESS} for {@code vertexInputs}, then for a given vertex buffer binding, any attribute data fetched <b>must</b> be entirely contained within the corresponding vertex buffer binding, as described in <a href="https://registry.khronos.org/vulkan/specs/latest/html/vkspec.html#fxvertex-input">Vertex Input Description</a></li>
     * <li>For each vertex attribute accessed by this command, if its {@link VkVertexInputAttributeDescription}{@code ::format} or {@link VkVertexInputAttributeDescription2EXT}{@code ::format} is a <a href="https://registry.khronos.org/vulkan/specs/latest/html/vkspec.html#formats-packed">packed format</a>, and <a href="https://registry.khronos.org/vulkan/specs/latest/html/vkspec.html#features-legacyVertexAttributes">{@code legacyVertexAttributes}</a> is not enabled, the value of {@code attribAddress}, calculated as described in <a href="https://registry.khronos.org/vulkan/specs/latest/html/vkspec.html#fxvertex-input-address-calculation">Vertex Input Calculation</a>, <b>must</b> be a multiple of the <a href="https://registry.khronos.org/vulkan/specs/latest/html/vkspec.html#formats">size of the {@code format}</a></li>
     * <li>For each vertex attribute accessed by this command, if its {@link VkVertexInputAttributeDescription}{@code ::format} or {@link VkVertexInputAttributeDescription2EXT}{@code ::format} is not a <a href="https://registry.khronos.org/vulkan/specs/latest/html/vkspec.html#formats-packed">packed format</a>, and either <a href="https://registry.khronos.org/vulkan/specs/latest/html/vkspec.html#features-legacyVertexAttributes">{@code legacyVertexAttributes}</a> is not enabled or {@code format} has 64-bit components, the value of {@code attribAddress}, calculated as described in <a href="https://registry.khronos.org/vulkan/specs/latest/html/vkspec.html#fxvertex-input-address-calculation">Vertex Input Calculation</a>, <b>must</b> be a multiple of the <a href="https://registry.khronos.org/vulkan/specs/latest/html/vkspec.html#formats">size of the components of the {@code format}</a></li>
     * <li>If there is a shader object bound to the {@link VK10#VK_SHADER_STAGE_VERTEX_BIT SHADER_STAGE_VERTEX_BIT} stage or the bound graphics pipeline state was created with the {@link VK13#VK_DYNAMIC_STATE_PRIMITIVE_TOPOLOGY DYNAMIC_STATE_PRIMITIVE_TOPOLOGY} dynamic state enabled then {@link VK13#vkCmdSetPrimitiveTopology CmdSetPrimitiveTopology} <b>must</b> have been called and not subsequently <a href="https://registry.khronos.org/vulkan/specs/latest/html/vkspec.html#dynamic-state-lifetime">invalidated</a> in the current command buffer prior to this drawing command</li>
     * <li>If the bound graphics pipeline state was created with the {@link VK13#VK_DYNAMIC_STATE_PRIMITIVE_TOPOLOGY DYNAMIC_STATE_PRIMITIVE_TOPOLOGY} dynamic state enabled and the <a href="https://registry.khronos.org/vulkan/specs/latest/html/vkspec.html#limits-dynamicPrimitiveTopologyUnrestricted">{@code dynamicPrimitiveTopologyUnrestricted}</a> is {@link VK10#VK_FALSE FALSE}, then the {@code primitiveTopology} parameter of {@code vkCmdSetPrimitiveTopology} <b>must</b> be of the same <a href="https://registry.khronos.org/vulkan/specs/latest/html/vkspec.html#drawing-primitive-topology-class">topology class</a> as the pipeline {@link VkPipelineInputAssemblyStateCreateInfo}{@code ::topology} state</li>
     * <li>If there is a shader object bound to the {@link VK10#VK_SHADER_STAGE_TESSELLATION_CONTROL_BIT SHADER_STAGE_TESSELLATION_CONTROL_BIT} stage, then {@link VK13#vkCmdSetPrimitiveTopology CmdSetPrimitiveTopology} <b>must</b> have set {@code primitiveTopology} to {@link VK10#VK_PRIMITIVE_TOPOLOGY_PATCH_LIST PRIMITIVE_TOPOLOGY_PATCH_LIST} prior to this drawing command</li>
     * <li>If the bound graphics pipeline was created with the {@link VK13#VK_DYNAMIC_STATE_VERTEX_INPUT_BINDING_STRIDE DYNAMIC_STATE_VERTEX_INPUT_BINDING_STRIDE} dynamic state enabled, but without the {@link EXTVertexInputDynamicState#VK_DYNAMIC_STATE_VERTEX_INPUT_EXT DYNAMIC_STATE_VERTEX_INPUT_EXT} dynamic state enabled, then {@link VK13#vkCmdBindVertexBuffers2 CmdBindVertexBuffers2} <b>must</b> have been called and not subsequently <a href="https://registry.khronos.org/vulkan/specs/latest/html/vkspec.html#dynamic-state-lifetime">invalidated</a> in the current command buffer prior to this draw command, and the {@code pStrides} parameter of {@link VK13#vkCmdBindVertexBuffers2 CmdBindVertexBuffers2} <b>must</b> not be {@code NULL}</li>
     * <li>If there is a shader object bound to the {@link VK10#VK_SHADER_STAGE_VERTEX_BIT SHADER_STAGE_VERTEX_BIT} stage or the bound graphics pipeline state was created with the {@link EXTVertexInputDynamicState#VK_DYNAMIC_STATE_VERTEX_INPUT_EXT DYNAMIC_STATE_VERTEX_INPUT_EXT} dynamic state enabled then {@link EXTShaderObject#vkCmdSetVertexInputEXT CmdSetVertexInputEXT} <b>must</b> have been called and not subsequently <a href="https://registry.khronos.org/vulkan/specs/latest/html/vkspec.html#dynamic-state-lifetime">invalidated</a> in the current command buffer prior to this draw command</li>
     * <li>If <a href="https://registry.khronos.org/vulkan/specs/latest/html/vkspec.html#features-vertexAttributeRobustness">{@code vertexAttributeRobustness}</a> is not enabled and there is a shader object bound to the {@link VK10#VK_SHADER_STAGE_VERTEX_BIT SHADER_STAGE_VERTEX_BIT} stage or the bound graphics pipeline state was created with the {@link EXTVertexInputDynamicState#VK_DYNAMIC_STATE_VERTEX_INPUT_EXT DYNAMIC_STATE_VERTEX_INPUT_EXT} dynamic state enabled then all variables with the {@code Input} storage class decorated with {@code Location} in the {@code Vertex} {@code Execution} {@code Model} {@code OpEntryPoint} <b>must</b> contain a location in {@link VkVertexInputAttributeDescription2EXT}{@code ::location}</li>
     * <li>If there is a shader object bound to the {@link VK10#VK_SHADER_STAGE_VERTEX_BIT SHADER_STAGE_VERTEX_BIT} stage or the bound graphics pipeline state was created with the {@link EXTVertexInputDynamicState#VK_DYNAMIC_STATE_VERTEX_INPUT_EXT DYNAMIC_STATE_VERTEX_INPUT_EXT} dynamic state enabled and either the <a href="https://registry.khronos.org/vulkan/specs/latest/html/vkspec.html#features-legacyVertexAttributes">{@code legacyVertexAttributes}</a> feature is not enabled or the SPIR-V Type associated with a given {@code Input} variable of the corresponding {@code Location} in the {@code Vertex} {@code Execution} {@code Model} {@code OpEntryPoint} is 64-bit, then the numeric type associated with all {@code Input} variables of the corresponding {@code Location} in the {@code Vertex} {@code Execution} {@code Model} {@code OpEntryPoint} <b>must</b> be the same as {@link VkVertexInputAttributeDescription2EXT}{@code ::format}</li>
     * <li>If there is a shader object bound to the {@link VK10#VK_SHADER_STAGE_VERTEX_BIT SHADER_STAGE_VERTEX_BIT} stage or the bound graphics pipeline state was created with the {@link EXTVertexInputDynamicState#VK_DYNAMIC_STATE_VERTEX_INPUT_EXT DYNAMIC_STATE_VERTEX_INPUT_EXT} dynamic state enabled and {@link VkVertexInputAttributeDescription2EXT}{@code ::format} has a 64-bit component, then the scalar width associated with all {@code Input} variables of the corresponding {@code Location} in the {@code Vertex} {@code Execution} {@code Model} {@code OpEntryPoint} <b>must</b> be 64-bit</li>
     * <li>If there is a shader object bound to the {@link VK10#VK_SHADER_STAGE_VERTEX_BIT SHADER_STAGE_VERTEX_BIT} stage or the bound graphics pipeline state was created with the {@link EXTVertexInputDynamicState#VK_DYNAMIC_STATE_VERTEX_INPUT_EXT DYNAMIC_STATE_VERTEX_INPUT_EXT} dynamic state enabled and the scalar width associated with a {@code Location} decorated {@code Input} variable in the {@code Vertex} {@code Execution} {@code Model} {@code OpEntryPoint} is 64-bit, then the corresponding {@link VkVertexInputAttributeDescription2EXT}{@code ::format} <b>must</b> have a 64-bit component</li>
     * <li>If there is a shader object bound to the {@link VK10#VK_SHADER_STAGE_VERTEX_BIT SHADER_STAGE_VERTEX_BIT} stage or the bound graphics pipeline state was created with the {@link EXTVertexInputDynamicState#VK_DYNAMIC_STATE_VERTEX_INPUT_EXT DYNAMIC_STATE_VERTEX_INPUT_EXT} dynamic state enabled and {@link VkVertexInputAttributeDescription2EXT}{@code ::format} has a 64-bit component, then all {@code Input} variables at the corresponding {@code Location} in the {@code Vertex} {@code Execution} {@code Model} {@code OpEntryPoint} <b>must</b> not use components that are not present in the format</li>
     * <li>If there is a shader object bound to the {@link VK10#VK_SHADER_STAGE_TESSELLATION_CONTROL_BIT SHADER_STAGE_TESSELLATION_CONTROL_BIT} stage or the bound graphics pipeline state was created with the {@link EXTExtendedDynamicState2#VK_DYNAMIC_STATE_PATCH_CONTROL_POINTS_EXT DYNAMIC_STATE_PATCH_CONTROL_POINTS_EXT} dynamic state enabled, and the <a href="https://registry.khronos.org/vulkan/specs/latest/html/vkspec.html#dynamic-state-current-value">current value</a> of {@code primitiveTopology} is {@link VK10#VK_PRIMITIVE_TOPOLOGY_PATCH_LIST PRIMITIVE_TOPOLOGY_PATCH_LIST}, then {@link EXTShaderObject#vkCmdSetPatchControlPointsEXT CmdSetPatchControlPointsEXT} <b>must</b> have been called and not subsequently <a href="https://registry.khronos.org/vulkan/specs/latest/html/vkspec.html#dynamic-state-lifetime">invalidated</a> in the current command buffer prior to this drawing command</li>
     * <li>If there is a shader object bound to the {@link VK10#VK_SHADER_STAGE_VERTEX_BIT SHADER_STAGE_VERTEX_BIT} stage or the bound graphics pipeline state was created with the {@link VK13#VK_DYNAMIC_STATE_PRIMITIVE_RESTART_ENABLE DYNAMIC_STATE_PRIMITIVE_RESTART_ENABLE} dynamic state enabled then {@link VK13#vkCmdSetPrimitiveRestartEnable CmdSetPrimitiveRestartEnable} <b>must</b> have been called and not subsequently <a href="https://registry.khronos.org/vulkan/specs/latest/html/vkspec.html#dynamic-state-lifetime">invalidated</a> in the current command buffer prior to this drawing command</li>
     * <li>If the <a href="https://registry.khronos.org/vulkan/specs/latest/html/vkspec.html#features-primitiveTopologyListRestart">{@code primitiveTopologyListRestart}</a> feature is not enabled, the topology is {@link VK10#VK_PRIMITIVE_TOPOLOGY_POINT_LIST PRIMITIVE_TOPOLOGY_POINT_LIST}, {@link VK10#VK_PRIMITIVE_TOPOLOGY_LINE_LIST PRIMITIVE_TOPOLOGY_LINE_LIST}, {@link VK10#VK_PRIMITIVE_TOPOLOGY_TRIANGLE_LIST PRIMITIVE_TOPOLOGY_TRIANGLE_LIST}, {@link VK10#VK_PRIMITIVE_TOPOLOGY_LINE_LIST_WITH_ADJACENCY PRIMITIVE_TOPOLOGY_LINE_LIST_WITH_ADJACENCY}, or {@link VK10#VK_PRIMITIVE_TOPOLOGY_TRIANGLE_LIST_WITH_ADJACENCY PRIMITIVE_TOPOLOGY_TRIANGLE_LIST_WITH_ADJACENCY}, there is a shader object bound to the {@link VK10#VK_SHADER_STAGE_VERTEX_BIT SHADER_STAGE_VERTEX_BIT} stage or the bound graphics pipeline state was created with the {@link VK13#VK_DYNAMIC_STATE_PRIMITIVE_RESTART_ENABLE DYNAMIC_STATE_PRIMITIVE_RESTART_ENABLE} dynamic state enabled then {@link VK13#vkCmdSetPrimitiveRestartEnable CmdSetPrimitiveRestartEnable} <b>must</b> be {@link VK10#VK_FALSE FALSE}</li>
     * <li>The bound graphics pipeline <b>must</b> not have been created with the {@link VkPipelineShaderStageCreateInfo}{@code ::stage} member of any element of {@link VkGraphicsPipelineCreateInfo}{@code ::pStages} set to {@link EXTMeshShader#VK_SHADER_STAGE_TASK_BIT_EXT SHADER_STAGE_TASK_BIT_EXT} or {@link EXTMeshShader#VK_SHADER_STAGE_MESH_BIT_EXT SHADER_STAGE_MESH_BIT_EXT}</li>
     * <li>There <b>must</b> be no shader object bound to either of the {@link EXTMeshShader#VK_SHADER_STAGE_TASK_BIT_EXT SHADER_STAGE_TASK_BIT_EXT} or {@link EXTMeshShader#VK_SHADER_STAGE_MESH_BIT_EXT SHADER_STAGE_MESH_BIT_EXT} stages</li>
     * <li>If a shader object is bound to the {@link VK10#VK_SHADER_STAGE_TESSELLATION_EVALUATION_BIT SHADER_STAGE_TESSELLATION_EVALUATION_BIT} stage or a graphics pipeline is bound which was created with the {@link EXTExtendedDynamicState3#VK_DYNAMIC_STATE_TESSELLATION_DOMAIN_ORIGIN_EXT DYNAMIC_STATE_TESSELLATION_DOMAIN_ORIGIN_EXT} dynamic state enabled, then {@link EXTShaderObject#vkCmdSetTessellationDomainOriginEXT CmdSetTessellationDomainOriginEXT} <b>must</b> have been called and not subsequently <a href="https://registry.khronos.org/vulkan/specs/latest/html/vkspec.html#dynamic-state-lifetime">invalidated</a> in the current command buffer prior to this drawing command</li>
     * <li>If there is no bound graphics pipeline, {@code vkCmdBindShadersEXT} <b>must</b> have been called in the current command buffer with {@code pStages} with an element of {@link VK10#VK_SHADER_STAGE_VERTEX_BIT SHADER_STAGE_VERTEX_BIT}</li>
     * <li>If there is no bound graphics pipeline, and the <a href="https://registry.khronos.org/vulkan/specs/latest/html/vkspec.html#features-tessellationShader">{@code tessellationShader}</a> feature is enabled, {@code vkCmdBindShadersEXT} <b>must</b> have been called in the current command buffer with {@code pStages} with an element of {@link VK10#VK_SHADER_STAGE_TESSELLATION_CONTROL_BIT SHADER_STAGE_TESSELLATION_CONTROL_BIT}</li>
     * <li>If there is no bound graphics pipeline, and the <a href="https://registry.khronos.org/vulkan/specs/latest/html/vkspec.html#features-tessellationShader">{@code tessellationShader}</a> feature is enabled, {@code vkCmdBindShadersEXT} <b>must</b> have been called in the current command buffer with {@code pStages} with an element of {@link VK10#VK_SHADER_STAGE_TESSELLATION_EVALUATION_BIT SHADER_STAGE_TESSELLATION_EVALUATION_BIT}</li>
     * <li>If there is no bound graphics pipeline, and the <a href="https://registry.khronos.org/vulkan/specs/latest/html/vkspec.html#features-geometryShader">{@code geometryShader}</a> feature is enabled, {@code vkCmdBindShadersEXT} <b>must</b> have been called in the current command buffer with {@code pStages} with an element of {@link VK10#VK_SHADER_STAGE_GEOMETRY_BIT SHADER_STAGE_GEOMETRY_BIT}</li>
     * </ul>
     * 
     * <ul>
     * <li>If the bound graphics pipeline state was created with {@link VkPipelineVertexInputDivisorStateCreateInfo} in the {@code pNext} chain of {@link VkGraphicsPipelineCreateInfo}{@code ::pVertexInputState}, any member of {@link VkPipelineVertexInputDivisorStateCreateInfo}{@code ::pVertexBindingDivisors} has a value other than 1 in {@code divisor}, and {@link VkPhysicalDeviceVertexAttributeDivisorProperties}{@code ::supportsNonZeroFirstInstance} is {@link VK10#VK_FALSE FALSE}, then {@code firstInstance} <b>must</b> be 0</li>
     * <li>If <a href="https://registry.khronos.org/vulkan/specs/latest/html/vkspec.html#shaders-objects">shader objects</a> are used for drawing or the bound graphics pipeline state was created with the {@link EXTVertexInputDynamicState#VK_DYNAMIC_STATE_VERTEX_INPUT_EXT DYNAMIC_STATE_VERTEX_INPUT_EXT} dynamic state enabled, any member of the {@code pVertexBindingDescriptions} parameter to the {@link EXTShaderObject#vkCmdSetVertexInputEXT CmdSetVertexInputEXT} call that sets this dynamic state has a value other than 1 in {@code divisor}, and {@link VkPhysicalDeviceVertexAttributeDivisorProperties}{@code ::supportsNonZeroFirstInstance} is {@link VK10#VK_FALSE FALSE}, then {@code firstInstance} <b>must</b> be 0</li>
     * </ul>
     * 
     * <ul>
     * <li>{@link VkPhysicalDeviceTransformFeedbackFeaturesEXT}{@code ::transformFeedback} <b>must</b> be enabled</li>
     * <li>The implementation <b>must</b> support {@link VkPhysicalDeviceTransformFeedbackPropertiesEXT}{@code ::transformFeedbackDraw}</li>
     * <li>{@code vertexStride} <b>must</b> be greater than 0 and less than or equal to {@link VkPhysicalDeviceTransformFeedbackPropertiesEXT}{@code ::maxTransformFeedbackBufferDataStride}</li>
     * <li>If {@code counterBuffer} is non-sparse then it <b>must</b> be bound completely and contiguously to a single {@code VkDeviceMemory} object</li>
     * <li>{@code counterBuffer} <b>must</b> have been created with the {@link VK10#VK_BUFFER_USAGE_INDIRECT_BUFFER_BIT BUFFER_USAGE_INDIRECT_BUFFER_BIT} bit set</li>
     * <li>{@code counterBufferOffset} <b>must</b> be a multiple of 4</li>
     * <li>{@code counterOffset} <b>must</b> be a multiple of 4</li>
     * <li>{@code vertexStride} <b>must</b> be a multiple of 4</li>
     * <li>{@code commandBuffer} <b>must</b> not be a protected command buffer</li>
     * </ul>
     * 
     * <h5>Valid Usage (Implicit)</h5>
     * 
     * <ul>
     * <li>{@code commandBuffer} <b>must</b> be a valid {@code VkCommandBuffer} handle</li>
     * <li>{@code counterBuffer} <b>must</b> be a valid {@code VkBuffer} handle</li>
     * <li>{@code commandBuffer} <b>must</b> be in the <a href="https://registry.khronos.org/vulkan/specs/latest/html/vkspec.html#commandbuffers-lifecycle">recording state</a></li>
     * <li>The {@code VkCommandPool} that {@code commandBuffer} was allocated from <b>must</b> support graphics operations</li>
     * <li>This command <b>must</b> only be called inside of a render pass instance</li>
     * <li>This command <b>must</b> only be called outside of a video coding scope</li>
     * <li>Both of {@code commandBuffer}, and {@code counterBuffer} <b>must</b> have been created, allocated, or retrieved from the same {@code VkDevice}</li>
     * </ul>
     * 
     * <h5>Host Synchronization</h5>
     * 
     * <ul>
     * <li>Host access to {@code commandBuffer} <b>must</b> be externally synchronized</li>
     * <li>Host access to the {@code VkCommandPool} that {@code commandBuffer} was allocated from <b>must</b> be externally synchronized</li>
     * </ul>
     * 
     * <h5>Command Properties</h5>
     * 
     * <table class="lwjgl">
     * <thead><tr><th><a href="https://registry.khronos.org/vulkan/specs/latest/html/vkspec.html#VkCommandBufferLevel">Command Buffer Levels</a></th><th><a href="https://registry.khronos.org/vulkan/specs/latest/html/vkspec.html#vkCmdBeginRenderPass">Render Pass Scope</a></th><th><a href="https://registry.khronos.org/vulkan/specs/latest/html/vkspec.html#vkCmdBeginVideoCodingKHR">Video Coding Scope</a></th><th><a href="https://registry.khronos.org/vulkan/specs/latest/html/vkspec.html#VkQueueFlagBits">Supported Queue Types</a></th><th><a href="https://registry.khronos.org/vulkan/specs/latest/html/vkspec.html#fundamentals-queueoperation-command-types">Command Type</a></th></tr></thead>
     * <tbody><tr><td>Primary Secondary</td><td>Inside</td><td>Outside</td><td>Graphics</td><td>Action</td></tr></tbody>
     * </table>
     *
     * @param commandBuffer       the command buffer into which the command is recorded.
     * @param instanceCount       the number of instances to draw.
     * @param firstInstance       the instance ID of the first instance to draw.
     * @param counterBuffer       the buffer handle from where the byte count is read.
     * @param counterBufferOffset the offset into the buffer used to read the byte count, which is used to calculate the vertex count for this draw call.
     * @param counterOffset       subtracted from the byte count read from the {@code counterBuffer} at the {@code counterBufferOffset}
     * @param vertexStride        the stride in bytes between each element of the vertex data that is used to calculate the vertex count from the counter value. This value is typically the same value that was used in the graphics pipeline state when the transform feedback was captured as the {@code XfbStride}.
     */
    public static void vkCmdDrawIndirectByteCountEXT(VkCommandBuffer commandBuffer, @NativeType("uint32_t") int instanceCount, @NativeType("uint32_t") int firstInstance, @NativeType("VkBuffer") long counterBuffer, @NativeType("VkDeviceSize") long counterBufferOffset, @NativeType("uint32_t") int counterOffset, @NativeType("uint32_t") int vertexStride) {
        long __functionAddress = commandBuffer.getCapabilities().vkCmdDrawIndirectByteCountEXT;
        if (CHECKS) {
            check(__functionAddress);
        }
        callPJJV(commandBuffer.address(), instanceCount, firstInstance, counterBuffer, counterBufferOffset, counterOffset, vertexStride, __functionAddress);
    }

    /** Array version of: {@link #vkCmdBindTransformFeedbackBuffersEXT CmdBindTransformFeedbackBuffersEXT} */
    public static void vkCmdBindTransformFeedbackBuffersEXT(VkCommandBuffer commandBuffer, @NativeType("uint32_t") int firstBinding, @NativeType("VkBuffer const *") long[] pBuffers, @NativeType("VkDeviceSize const *") long[] pOffsets, @NativeType("VkDeviceSize const *") long @Nullable [] pSizes) {
        long __functionAddress = commandBuffer.getCapabilities().vkCmdBindTransformFeedbackBuffersEXT;
        if (CHECKS) {
            check(__functionAddress);
            check(pOffsets, pBuffers.length);
            checkSafe(pSizes, pBuffers.length);
        }
        callPPPPV(commandBuffer.address(), firstBinding, pBuffers.length, pBuffers, pOffsets, pSizes, __functionAddress);
    }

    /** Array version of: {@link #vkCmdBeginTransformFeedbackEXT CmdBeginTransformFeedbackEXT} */
    public static void vkCmdBeginTransformFeedbackEXT(VkCommandBuffer commandBuffer, @NativeType("uint32_t") int firstCounterBuffer, @NativeType("VkBuffer const *") long @Nullable [] pCounterBuffers, @NativeType("VkDeviceSize const *") long @Nullable [] pCounterBufferOffsets) {
        long __functionAddress = commandBuffer.getCapabilities().vkCmdBeginTransformFeedbackEXT;
        if (CHECKS) {
            check(__functionAddress);
            checkSafe(pCounterBufferOffsets, lengthSafe(pCounterBuffers));
        }
        callPPPV(commandBuffer.address(), firstCounterBuffer, lengthSafe(pCounterBuffers), pCounterBuffers, pCounterBufferOffsets, __functionAddress);
    }

    /** Array version of: {@link #vkCmdEndTransformFeedbackEXT CmdEndTransformFeedbackEXT} */
    public static void vkCmdEndTransformFeedbackEXT(VkCommandBuffer commandBuffer, @NativeType("uint32_t") int firstCounterBuffer, @NativeType("VkBuffer const *") long @Nullable [] pCounterBuffers, @NativeType("VkDeviceSize const *") long @Nullable [] pCounterBufferOffsets) {
        long __functionAddress = commandBuffer.getCapabilities().vkCmdEndTransformFeedbackEXT;
        if (CHECKS) {
            check(__functionAddress);
            checkSafe(pCounterBufferOffsets, lengthSafe(pCounterBuffers));
        }
        callPPPV(commandBuffer.address(), firstCounterBuffer, lengthSafe(pCounterBuffers), pCounterBuffers, pCounterBufferOffsets, __functionAddress);
    }

}