/*
 * Copyright LWJGL. All rights reserved.
 * License terms: https://www.lwjgl.org/license
 */
package par

import org.lwjgl.generator.*

// par_octasphere

val par_octasphere_uv_mode = "par_octasphere_uv_mode".enumType
val par_octasphere_normals_mode = "par_octasphere_normals_mode".enumType

val par_octasphere_config = struct(Module.PAR, "ParOctasphereConfig", nativeName = "par_octasphere_config") {
    float("corner_radius", "")
    float("width", "")
    float("height", "")
    float("depth", "")
    int("num_subdivisions", "")
    par_octasphere_uv_mode("uv_mode", "")
    par_octasphere_normals_mode("normals_mode", "")
}

val par_octasphere_mesh = struct(Module.PAR, "ParOctasphereMesh", nativeName = "par_octasphere_mesh", mutable = false) {
    float.p("positions", "").mutable()
    nullable..float.p("normals", "").mutable()
    nullable..float.p("texcoords", "").mutable()
    uint16_t.p("indices", "").mutable()
    uint32_t("num_indices", "")
    uint32_t("num_vertices", "")
}

// par_shapes

val PAR_SHAPES_T = typedef(uint32_t, "PAR_SHAPES_T")

val par_shapes_mesh = struct(Module.PAR, "ParShapesMesh", nativeName = "par_shapes_mesh", mutable = false) {
    documentation =
        """
        The {@code par_shapes.h} mesh structure.

        <b>LWJGL note</b>: in the default build, {@code PAR_SHAPES_T} is {@code uint32_t}.
        """

    float.p("points", "a flat list of 3-tuples (X Y Z X Y Z...) for the vertex positions")
    int("npoints", "the number of points")
    PAR_SHAPES_T.p("triangles", "flat list of 3-tuples (I J K I J K...)")
    int("ntriangles", "the number of triangles")
    nullable..float.p("normals", "an optional list of 3-tuples (X Y Z X Y Z...) for the vertex normals")
    nullable..float.p("tcoords", "an optional list of 2-tuples (U V U V U V...) for the vertex texture coordinates")
}

val par_shapes_rand_fn = Module.PAR.callback {
    float(
        "ParShapesRandFn",
        "",

        nullable..opaque_p("context", ""),

        nativeType = "par_shapes_rand_fn"
    ) {
        documentation = "Instances of this interface may be passed to the #create_lsystem() method."
    }
}

// par_streamlines

val parsl_annotation = struct(Module.PAR, "ParSLAnnotation", nativeName = "parsl_annotation", mutable = false) {
    documentation = "Layout for generated vertex attributes."

    float("u_along_curve", "longitudinal coordinate (see {@link ParStreamlines\\#PAR_U_MODE_NORMALIZED_DISTANCE parsl_u_mode})")
    float("v_across_curve", "either + or - depending on the side")
    float("spine_to_edge_x", "normalized vector from spine to edge")
    float("spine_to_edge_y", "normalized vector from spine to edge")
}

val parsl_position = struct(Module.PAR, "ParSLPosition", nativeName = "parsl_position") {
    documentation = "The simple two-tuple math type used for mesh and spine vertices."

    float("x", "")
    float("y", "")
}

val parsl_mesh = struct(Module.PAR, "ParSLMesh", nativeName = "parsl_mesh", mutable = false) {
    documentation =
        """
        Triangle mesh generated by the library.
        
        The vertex data is owned by streamlines context and becomes invalid on any subsequent call to the API. The {@code annotations}, {@code spine_lengths},
        and {@code random_offsets} fields are #NULL unless their corresponding flags have been set in ##ParSLConfig. 
        """

    AutoSize(
        "positions",
        "annotations",
        "spine_lengths",
        "random_offsets"
    )..uint32_t("num_vertices", "")
    uint32_t("num_triangles", "")
    uint32_t.p("triangle_indices", "")
    parsl_position.p("positions", "")
    nullable..parsl_annotation.p("annotations", "")
    nullable..float.p("spine_lengths", "")
    nullable..float.p("random_offsets", "")
}

val parsl_viewport = struct(Module.PAR, "ParSLViewport", nativeName = "parsl_viewport") {
    documentation = "Viewport for streamline seed placement."

    float("left", "")
    float("top", "")
    float("right", "")
    float("bottom", "")
}

val parsl_config = struct(Module.PAR, "ParSLConfig", nativeName = "parsl_config") {
    documentation = "Immutable configuration for a streamlines context."

    float("thickness", "")
    uint32_t("flags", "").links("FLAG_\\w+", LinkMode.BITFIELD)
    int("u_mode", "").links("PAR_U_MODE_\\w+")
    float("curves_max_flatness", "")
    float("streamlines_seed_spacing", "")
    parsl_viewport("streamlines_seed_viewport", "")
    float("miter_limit", "")
}

val parsl_spine_list = struct(Module.PAR, "ParSLSpineList", nativeName = "parsl_spine_list") {
    documentation = "Client-owned list of line strips that will be tessellated."

    AutoSize("vertices")..uint32_t("num_vertices", "")
    AutoSize("spine_lengths")..uint16_t("num_spines", "")
    parsl_position.p("vertices", "")
    uint16_t.p("spine_lengths", "")
    bool("closed", "")
}

val parsl_context = "parsl_context".opaque

val parsl_advection_callback = Module.PAR.callback {
    void(
        "ParSLAdvectionCallback",
        "Client function that moves a streamline particle by a single time step.",

        parsl_position.p("point", ""),
        opaque_p("userdata", ""),

        nativeType = "parsl_advection_callback"
    ) {
        documentation = "Instances of this interface may be passed to the #mesh_from_streamlines() method."
    }
}