/*
 * Copyright LWJGL. All rights reserved.
 * License terms: https://www.lwjgl.org/license
 */
package opengl.templates

import org.lwjgl.generator.*
import opengl.*

val ARB_direct_state_access = "ARBDirectStateAccess".nativeClassGL("ARB_direct_state_access") {
    documentation =
        """
        Native bindings to the $registryLink extension.

        In unextended OpenGL, most mutation of state contained in objects is through an indirection known as a binding. Objects are attached to a context
        (either directly or indirectly via a container) and then commands to modify or query their state are issued on that context, indirecting through its
        attachments and into the underlying object. This is known as `bind-to-edit'.

        This extension derives from the GL_EXT_direct_state_access extension, which added accessors for most state on most objects, allowing it to be queried
        and modified without the object needing to be bound to a context. In cases where a single property of an object is to be modified, directly accessing
        its state can be more efficient than binding the object to the context and then indirecting through it. Further, directly accessing the state of objects
        through their names rather than by bind-to-edit does not disturb the bindings of the current context, which is useful for tools, middleware and other
        applications that are unaware of the outer state but it can also avoid cases of redundant state changes.

        Requires ${GL20.core}. ${GL45C.promoted}
        """

    IntConstant(
        "Accepted by the {@code pname} parameter of GetTextureParameter{if}v and GetTextureParameterI{i ui}v.",

        "TEXTURE_TARGET"..0x1006
    )

    IntConstant(
        "Accepted by the {@code pname} parameter of GetQueryObjectiv.",

        "QUERY_TARGET"..0x82EA
    )

    fun dsa(extension: String, func: String) {
        DependsOn("$extension(ext)")..(reuse(GL45C, func))
    }

    dsa("ARB_transform_feedback2", "CreateTransformFeedbacks")
    dsa("ARB_uniform_buffer_object", "TransformFeedbackBufferBase") // TODO: interaction not mentioned, spec bug?
    dsa("ARB_uniform_buffer_object", "TransformFeedbackBufferRange") // TODO: interaction not mentioned, spec bug?
    dsa("ARB_transform_feedback2", "GetTransformFeedbackiv")
    dsa("ARB_transform_feedback2", "GetTransformFeedbacki_v")
    dsa("ARB_transform_feedback2", "GetTransformFeedbacki64_v")
    reuse(GL45C, "CreateBuffers")
    dsa("ARB_buffer_storage", "NamedBufferStorage")
    reuse(GL45C, "NamedBufferData")
    reuse(GL45C, "NamedBufferSubData")
    dsa("ARB_copy_buffer", "CopyNamedBufferSubData")
    dsa("ARB_clear_texture", "ClearNamedBufferData")
    dsa("ARB_clear_texture", "ClearNamedBufferSubData")
    reuse(GL45C, "MapNamedBuffer")
    dsa("ARB_map_buffer_range", "MapNamedBufferRange")
    reuse(GL45C, "UnmapNamedBuffer")
    dsa("ARB_map_buffer_range", "FlushMappedNamedBufferRange")
    reuse(GL45C, "GetNamedBufferParameteriv")
    reuse(GL45C, "GetNamedBufferParameteri64v")
    reuse(GL45C, "GetNamedBufferPointerv")
    reuse(GL45C, "GetNamedBufferSubData")
    dsa("ARB_framebuffer_object", "CreateFramebuffers")
    dsa("ARB_framebuffer_object", "NamedFramebufferRenderbuffer")
    dsa("ARB_framebuffer_no_attachments", "NamedFramebufferParameteri")
    dsa("ARB_framebuffer_object", "NamedFramebufferTexture")
    dsa("ARB_framebuffer_object", "NamedFramebufferTextureLayer")
    dsa("ARB_framebuffer_object", "NamedFramebufferDrawBuffer")
    dsa("ARB_framebuffer_object", "NamedFramebufferDrawBuffers")
    dsa("ARB_framebuffer_object", "NamedFramebufferReadBuffer")
    dsa("ARB_invalidate_subdata", "InvalidateNamedFramebufferData")
    dsa("ARB_invalidate_subdata", "InvalidateNamedFramebufferSubData")
    dsa("ARB_framebuffer_object", "ClearNamedFramebufferiv")
    dsa("ARB_framebuffer_object", "ClearNamedFramebufferuiv")
    dsa("ARB_framebuffer_object", "ClearNamedFramebufferfv")
    dsa("ARB_framebuffer_object", "ClearNamedFramebufferfi")
    dsa("ARB_framebuffer_object", "BlitNamedFramebuffer")
    dsa("ARB_framebuffer_object", "CheckNamedFramebufferStatus")
    dsa("ARB_framebuffer_no_attachments", "GetNamedFramebufferParameteriv")
    dsa("ARB_framebuffer_object", "GetNamedFramebufferAttachmentParameteriv")
    dsa("ARB_framebuffer_object", "CreateRenderbuffers")
    dsa("ARB_framebuffer_object", "NamedRenderbufferStorage")
    dsa("ARB_framebuffer_object", "NamedRenderbufferStorageMultisample")
    dsa("ARB_framebuffer_object", "GetNamedRenderbufferParameteriv")
    reuse(GL45C, "CreateTextures")
    dsa("ARB_texture_buffer_object", "TextureBuffer") // TODO: TextureBuffer depends on ARB_transform_feedback2? spec bug?
    dsa("ARB_texture_buffer_range", "TextureBufferRange")
    dsa("ARB_texture_storage", "TextureStorage1D")
    dsa("ARB_texture_storage", "TextureStorage2D")
    dsa("ARB_texture_storage", "TextureStorage3D")
    dsa("ARB_texture_storage_multisample", "TextureStorage2DMultisample")
    dsa("ARB_texture_storage_multisample", "TextureStorage3DMultisample")
    reuse(GL45C, "TextureSubImage1D")
    reuse(GL45C, "TextureSubImage2D")
    reuse(GL45C, "TextureSubImage3D")
    reuse(GL45C, "CompressedTextureSubImage1D")
    reuse(GL45C, "CompressedTextureSubImage2D")
    reuse(GL45C, "CompressedTextureSubImage3D")
    reuse(GL45C, "CopyTextureSubImage1D")
    reuse(GL45C, "CopyTextureSubImage2D")
    reuse(GL45C, "CopyTextureSubImage3D")
    reuse(GL45C, "TextureParameterf")
    reuse(GL45C, "TextureParameterfv")
    reuse(GL45C, "TextureParameteri")
    reuse(GL45C, "TextureParameterIiv")
    reuse(GL45C, "TextureParameterIuiv")
    reuse(GL45C, "TextureParameteriv")
    dsa("ARB_framebuffer_object", "GenerateTextureMipmap")
    reuse(GL45C, "BindTextureUnit")
    reuse(GL45C, "GetTextureImage")
    reuse(GL45C, "GetCompressedTextureImage")
    reuse(GL45C, "GetTextureLevelParameterfv")
    reuse(GL45C, "GetTextureLevelParameteriv")
    reuse(GL45C, "GetTextureParameterfv")
    reuse(GL45C, "GetTextureParameterIiv")
    reuse(GL45C, "GetTextureParameterIuiv")
    reuse(GL45C, "GetTextureParameteriv")
    dsa("ARB_vertex_array_object", "CreateVertexArrays")
    dsa("ARB_vertex_array_object", "DisableVertexArrayAttrib")
    dsa("ARB_vertex_array_object", "EnableVertexArrayAttrib")
    dsa("ARB_vertex_array_object", "VertexArrayElementBuffer")
    dsa("ARB_vertex_attrib_binding", "VertexArrayVertexBuffer")
    dsa("ARB_multi_bind", "VertexArrayVertexBuffers")
    dsa("ARB_vertex_attrib_binding", "VertexArrayAttribFormat")
    dsa("ARB_vertex_attrib_binding", "VertexArrayAttribIFormat")
    dsa("ARB_vertex_attrib_binding", "VertexArrayAttribLFormat")
    dsa("ARB_vertex_attrib_binding", "VertexArrayAttribBinding")
    dsa("ARB_vertex_attrib_binding", "VertexArrayBindingDivisor")
    dsa("ARB_vertex_array_object", "GetVertexArrayiv")
    dsa("ARB_vertex_array_object", "GetVertexArrayIndexediv")
    dsa("ARB_vertex_array_object", "GetVertexArrayIndexed64iv")
    dsa("ARB_sampler_objects", "CreateSamplers")
    dsa("ARB_separate_shader_objects", "CreateProgramPipelines")
    reuse(GL45C, "CreateQueries")
    dsa("ARB_query_buffer_object", "GetQueryBufferObjecti64v")
    dsa("ARB_query_buffer_object", "GetQueryBufferObjectiv")
    dsa("ARB_query_buffer_object", "GetQueryBufferObjectui64v")
    dsa("ARB_query_buffer_object", "GetQueryBufferObjectuiv")
}